<?php

namespace Vtlabs\Event\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Event;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Event\Models\EventArtist;
use Vtlabs\Event\Filters\EventFilter;
use Vtlabs\Event\Models\EventCategory;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Event\Http\Resources\Admin\EventAdminResource;

class EventController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'title' => 'sometimes|string'
        ]);

        $events = Event::filter($request->all(), EventFilter::class);

        return EventAdminResource::collection($events->paginate());
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'short_description_translations' => 'sometimes|translation',
            'description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'auditorium_id' => 'required|exists:ev_auditoriums,id',
            'date' => 'required|date_format:Y-m-d',
            'time_from' => 'required',
            'time_to' => 'required',
            'area' => 'sometimes|string',
            'address' => 'required|string',
            'longitude' => 'required|numeric|min:-180|max:180',
            'latitude' => 'required|numeric|min:-90|max:90',
            'event_categories' => 'required|array',
            'event_categories.*.id' => 'required|exists:categories,id',
            'event_categories.*.price' => 'required|numeric',
            'event_artists' => 'required|array',
            'event_artists.*.id' => 'required|exists:ev_artists,id',
            'event_artists.*.time_from' => 'required',
            'event_artists.*.time_to' => 'required',
            'image' => 'sometimes|image',
            'is_vip' => 'sometimes|boolean'
        ]);

        CoreHelper::requestMergeJsonField();

        $event = Event::create($request->only([
            'title', 'short_description', 'description', 'meta', 'date', 'time_from', 'time_to', 'area', 'address',
            'longitude', 'latitude', 'auditorium_id', 'is_vip'
        ]));

        if ($request->image) {
            $event->addMedia($request->image)->toMediaCollection("images");
        }

        // attach categories
        foreach ($request->event_categories as $eventCategory) {
            EventCategory::create([
                'category_id' => $eventCategory['id'],
                'price' => $eventCategory['price'],
                'event_id' => $event->id
            ]);
        }

        // attach artists
        foreach ($request->event_artists as $eventArtist) {
            EventArtist::create([
                'artist_id' => $eventArtist['id'],
                'time_from' => $eventArtist['time_from'],
                'time_to' => $eventArtist['time_to'],
                'event_id' => $event->id
            ]);
        }

        return new EventAdminResource($event->fresh());
    }

    public function show(Event $event)
    {
        return new EventAdminResource($event);
    }

    public function update(Event $event, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'short_description_translations' => 'sometimes|translation',
            'description_translations' => 'sometimes|translation',
            'meta' => 'sometimes|json|nullable',
            'auditorium_id' => 'required|exists:ev_auditoriums,id',
            'date' => 'required|date_format:Y-m-d',
            'time_from' => 'required',
            'time_to' => 'required',
            'area' => 'sometimes|string',
            'address' => 'required|string',
            'longitude' => 'required|numeric|min:-180|max:180',
            'latitude' => 'required|numeric|min:-90|max:90',
            'event_categories' => 'required|array',
            'event_categories.*.id' => 'required|exists:categories,id',
            'event_categories.*.price' => 'required|numeric',
            'event_artists' => 'required|array',
            'event_artists.*.id' => 'required|exists:ev_artists,id',
            'event_artists.*.time_from' => 'required',
            'event_artists.*.time_to' => 'required',
            'image' => 'sometimes|image',
            'is_vip' => 'sometimes|boolean'
        ]);

        CoreHelper::requestMergeJsonField();

        $event->fill($request->only([
            'title', 'short_description', 'description', 'meta', 'date', 'time_from', 'time_to', 'area', 'address',
            'longitude', 'latitude', 'auditorium_id', 'is_vip'
        ]));
        $event->save();

        if ($request->image) {
            $event->clearMediaCollection('images');
            $event->addMedia($request->image)->toMediaCollection("images");
        }

        // attach categories
        EventCategory::where('event_id', $event->id)->delete();
        foreach ($request->event_categories as $eventCategory) {
            EventCategory::create([
                'category_id' => $eventCategory['id'],
                'price' => $eventCategory['price'],
                'event_id' => $event->id
            ]);
        }

        // attach artists

        EventArtist::where('event_id', $event->id)->delete();
        foreach ($request->event_artists as $eventArtist) {
            EventArtist::create([
                'artist_id' => $eventArtist['id'],
                'time_from' => $eventArtist['time_from'],
                'time_to' => $eventArtist['time_to'],
                'event_id' => $event->id
            ]);
        }


        return new EventAdminResource($event->fresh());
    }

    public function destroy(Event $event)
    {
        Gate::authorize('delete');

        $event->delete();

        return response()->json([], 200);
    }
}
