<?php

namespace Vtlabs\Event\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Seat;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Event\Models\Auditorium;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Event\Filters\AuditoriumFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Category\Http\Resources\CategoryResource;
use Vtlabs\Event\Http\Resources\Admin\AuditoriumAdminResource;

class AuditoriumController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string',
            'venue' => 'sometimes|exists:ev_venues,id'
        ]);

        $auditoriums = Auditorium::filter($request->all(), AuditoriumFilter::class);

        if ($request->pagination == '0') {
            $auditoriums = $auditoriums->get();
        } else {
            $auditoriums = $auditoriums->paginate();
        }

        return AuditoriumAdminResource::collection($auditoriums);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'venue_id' => 'required|exists:ev_venues,id',
            'seats' => 'required|array',
            'seats.*.min_row' => 'sometimes|regex:/[A-Z]/|nullable', // required if seats.type = seat
            'seats.*.max_row' => 'sometimes|regex:/[A-Z]/|nullable',
            'seats.*.min_col' => 'sometimes|nullable',
            'seats.*.max_col' => 'sometimes|nullable',
            'seats.*.category_id' => 'required|exists:categories,id',
            'seats.*.type' => 'required|in:standing,seat',
            'seats.*.standing_count' => 'sometimes|numeric|nullable',
        ]);

        CoreHelper::requestMergeJsonField();

        $auditorium = Auditorium::create($request->only(['title', 'meta', 'venue_id']));

        // generate seats
        foreach ($request->seats as $seat) {
            $categoryId = $seat['category_id'];
            $type = $seat['type'];

            if ($type == 'seat') {
                $rowMin = strtoupper($seat['min_row']);
                $rowMax = strtoupper($seat['max_row']);
                $colMin = $seat['min_col'];
                $colMax = $seat['max_col'];

                for ($i = ord($rowMin); $i <= ord($rowMax); $i++) {
                    for ($j = $colMin; $j <= $colMax; $j++) {
                        Seat::insert([
                            'row' => chr($i),
                            'column' => $j,
                            'auditorium_id' => $auditorium->id,
                            'category_id' => $categoryId,
                            'type' => $type
                        ]);
                    }
                }
            } else if ($type == 'standing' && $seat['standing_count'] > 0) {
                for ($i = 0; $i < $seat['standing_count']; $i++) {
                    Seat::insert([
                        'category_id' => $categoryId,
                        'auditorium_id' => $auditorium->id,
                        'type' => $type
                    ]);
                }
            }
        }

        return new AuditoriumAdminResource($auditorium->fresh());
    }

    public function show(Auditorium $auditorium)
    {
        return new AuditoriumAdminResource($auditorium);
    }

    public function update(Auditorium $auditorium, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'venue_id' => 'required|exists:ev_venues,id',
            'seats' => 'sometimes|array',
            'seats.*.min_row' => 'sometimes|regex:/[A-Z]/|nullable', // required if seats.type = seat
            'seats.*.max_row' => 'sometimes|regex:/[A-Z]/|nullable',
            'seats.*.min_col' => 'sometimes|nullable',
            'seats.*.max_col' => 'sometimes|nullable',
            'seats.*.category_id' => 'required|exists:categories,id',
            'seats.*.type' => 'required|in:standing,seat',
            'seats.*.standing_count' => 'sometimes|numeric|nullable',
        ]);

        CoreHelper::requestMergeJsonField();

        $auditorium->fill($request->only(['title', 'meta', 'venue_id']));
        $auditorium->save();

        // generate seats
        foreach ($request->seats as $seat) {
            Seat::where('auditorium_id', $auditorium->id)->delete(); // delete existing seats
            $categoryId = $seat['category_id'];
            $type = $seat['type'];

            if ($type == 'seat') {
                $rowMin = strtoupper($seat['min_row']);
                $rowMax = strtoupper($seat['max_row']);
                $colMin = $seat['min_col'];
                $colMax = $seat['max_col'];

                for ($i = ord($rowMin); $i <= ord($rowMax); $i++) {
                    for ($j = $colMin; $j <= $colMax; $j++) {
                        Seat::insert([
                            'row' => chr($i),
                            'column' => $j,
                            'auditorium_id' => $auditorium->id,
                            'category_id' => $categoryId,
                            'type' => $type
                        ]);
                    }
                }
            } else if ($type == 'standing' && $seat['standing_count'] > 0) {
                for ($i = 0; $i < $seat['standing_count']; $i++) {
                    Seat::insert([
                        'category_id' => $categoryId,
                        'auditorium_id' => $auditorium->id,
                        'type' => $type
                    ]);
                }
            }
        }

        return new AuditoriumAdminResource($auditorium->fresh());
    }

    public function destroy(Auditorium $auditorium)
    {
        Gate::authorize('delete');

        $auditorium->delete();

        return response()->json([], 200);
    }

    public function categories(Auditorium $auditorium)
    {
        // all the categories this event's auditoriums seats have
        $categories = [];
        $distinctCategoriesOfSeats =  $auditorium->seats->unique('category_id');
        foreach ($distinctCategoriesOfSeats as $seat) {
            $categories[] = $seat->category;
        }

        return CategoryResource::collection($categories);
    }
}
