<?php

namespace Vtlabs\Event\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Event\Models\Seat;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Event\Models\Auditorium;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Event\Filters\AuditoriumFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Category\Http\Resources\CategoryResource;
use Vtlabs\Event\Http\Resources\Admin\AuditoriumAdminResource;

class AuditoriumController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'title' => 'sometimes|string',
            'venue' => 'sometimes|exists:ev_venues,id'
        ]);

        $auditoriums = Auditorium::filter($request->all(), AuditoriumFilter::class);

        if ($request->pagination == '0') {
            $auditoriums = $auditoriums->get();
        } else {
            $auditoriums = $auditoriums->paginate();
        }

        return AuditoriumAdminResource::collection($auditoriums);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'venue_id' => 'required|exists:ev_venues,id',
            'seats' => 'required|array',
            'seats.*.min_row' => 'required|regex:/[A-Z]/',
            'seats.*.max_row' => 'required|regex:/[A-Z]/',
            'seats.*.min_col' => 'required',
            'seats.*.max_col' => 'required',
            'seats.*.category_id' => 'required|exists:categories,id',
        ]);

        CoreHelper::requestMergeJsonField();

        $auditorium = Auditorium::create($request->only(['title', 'meta', 'venue_id']));

        // generate seats
        foreach ($request->seats as $seat) {
            $rowMin = strtoupper($seat['min_row']);
            $rowMax = strtoupper($seat['max_row']);
            $colMin = $seat['min_col'];
            $colMax = $seat['max_col'];
            $categoryId = $seat['category_id'];

            for ($i = ord($rowMin); $i <= ord($rowMax); $i++) {
                for ($j = $colMin; $j <= $colMax; $j++) {
                    Seat::insert([
                        'row' => chr($i),
                        'column' => $j,
                        'auditorium_id' => $auditorium->id,
                        'category_id' => $categoryId
                    ]);
                }
            }
        }

        return new AuditoriumAdminResource($auditorium->fresh());
    }

    public function show(Auditorium $auditorium)
    {
        return new AuditoriumAdminResource($auditorium);
    }

    public function update(Auditorium $auditorium, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'venue_id' => 'required|exists:ev_venues,id',
            'seats' => 'sometimes|array',
            'seats.*.min_row' => 'required|regex:/[A-Z]/',
            'seats.*.max_row' => 'required|regex:/[A-Z]/',
            'seats.*.min_col' => 'required',
            'seats.*.max_col' => 'required',
            'seats.*.category_id' => 'required|exists:categories,id',
        ]);

        CoreHelper::requestMergeJsonField();

        $auditorium->fill($request->only(['title', 'meta', 'venue_id']));
        $auditorium->save();

        // generate seats
        if ($request->seats) {
            Seat::where('auditorium_id', $auditorium->id)->delete(); // delete existing seats
            foreach ($request->seats as $seat) {
                $rowMin = strtoupper($seat['min_row']);
                $rowMax = strtoupper($seat['max_row']);
                $colMin = $seat['min_col'];
                $colMax = $seat['max_col'];
                $categoryId = $seat['category_id'];

                for ($i = ord($rowMin); $i <= ord($rowMax); $i++) {
                    for ($j = $colMin; $j <= $colMax; $j++) {
                        Seat::insert([
                            'row' => chr($i),
                            'column' => $j,
                            'auditorium_id' => $auditorium->id,
                            'category_id' => $categoryId
                        ]);
                    }
                }
            }
        }

        return new AuditoriumAdminResource($auditorium->fresh());
    }

    public function destroy(Auditorium $auditorium)
    {
        Gate::authorize('delete');

        $auditorium->delete();

        return response()->json([], 200);
    }

    public function categories(Auditorium $auditorium)
    {
        // all the categories this event's auditoriums seats have
        $categories = [];
        $distinctCategoriesOfSeats =  $auditorium->seats->unique('category_id');
        foreach ($distinctCategoriesOfSeats as $seat) {
            $categories[] = $seat->category;
        }

        return CategoryResource::collection($categories);
    }
}
