<?php

namespace Vtlabs\Ecommerce\Models;

use EloquentFilter\Filterable;
use Spatie\MediaLibrary\HasMedia;
use Vtlabs\Payment\Traits\CanPay;
use Illuminate\Support\Facades\App;
use Rennokki\Plans\Traits\HasPlans;
use Spatie\ModelStatus\HasStatuses;
use Vtlabs\Core\Helpers\CoreHelper;
use Rennokki\Rating\Traits\CanBeRated;
use Illuminate\Database\Eloquent\Model;
use Rennokki\Rating\Contracts\Rateable;
use Spatie\Translatable\HasTranslations;
use Vtlabs\Core\Traits\CoreHasMediaTrait;
use Vtlabs\Appointment\Contracts\Appointee;
use Vtlabs\Appointment\Traits\CanBeAppointed;
use ChristianKuri\LaravelFavorite\Traits\Favoriteable;

class Vendor extends Model implements HasMedia, Rateable, Appointee
{
    use Filterable,
        Favoriteable,
        HasStatuses,
        HasTranslations,
        CoreHasMediaTrait,
        CanBeRated,
        CanBeAppointed,
        HasPlans,
        CanPay;

    protected $table = 'ecommerce_vendors';

    protected $guarded = [];

    protected $casts = [
        'name' => 'json',
        'details' => 'json',
        'tagline' => 'json',
        'meta' => 'json',
        'minimum_order' => 'integer',
        'delivery_fee' => 'float',
        'longitude' => 'float',
        'latitude' => 'float',
        'orders_count' => 'integer',
        'average_ratings' => 'float',
    ];

    protected $appends = ['mediaurls'];

    public $translatable = ['name', 'details', 'tagline'];

    protected $with = ['user', 'media', 'categories', 'productCategories'];

    public static function boot()
    {
        parent::boot();

        // on create
        static::created(function ($vendor) {
            $vendor->setStatus(config('vtlabs_ecommerce.vendor.status_default'));
        });
    }

    public static function findByUser($userId)
    {
        return Vendor::where('user_id', $userId)->firstOrFail();
    }

    public static function syncAdminProducts(Vendor $vendor)
    {
        $vendorQuery = Vendor::query();
        CoreHelper::searchTranslated($vendorQuery, 'name', App::getLocale(), "Admin Store");
        $adminVendor = $vendorQuery->oldest()->firstOrFail();

        $adminVendorProducts = $adminVendor->vendorProducts()->whereHas('product.categories', function($query) use ($vendor) {
            return $query->whereIn('id', $vendor->categories->pluck('id')->toArray());
        })->get();
        
        foreach ($adminVendorProducts as $adminVendorProduct) {
            $productCategories = $adminVendorProduct->product->categories->pluck('id')->toArray();
            
            // clone a product
            $newProduct = Product::create($adminVendorProduct->product->only(['title', 'detail', 'meta', 'price', 'video_url']));

            // sync categories
            $newProduct->categories()->sync($productCategories);

            $vendorProductCategories = [];
            for ($i = 0; $i < count($productCategories); $i++) {
                $vendorProductCategories[$productCategories[$i]] = ['product_id' => $newProduct->id];
            }
            $vendor->productCategories()->wherePivot('product_id', $newProduct->id)->sync($vendorProductCategories);

            // handle vendor product
            $vendorProductAttributes = array_merge($adminVendorProduct->only([
                'price', 'sale_price', 'sale_price_from', 'sale_price_to',
                'stock_quantity', 'stock_low_threshold'
            ]), ["product_id" => $newProduct->id, 'vendor_id' => $vendor->id]);

            // create vendor specific details of product
            VendorProduct::create($vendorProductAttributes);

            // handle addons
            foreach($adminVendorProduct->product->addonGroups as $addOnGroup) {
                $newAddOnGroup = $addOnGroup->replicate();
                $newAddOnGroup->product_id = $newProduct->id;
                $newAddOnGroup->save();

                foreach ($addOnGroup->addonChoices as $addOnChoice) {
                    $newAddOnItem = $addOnChoice->replicate();
                    $newAddOnItem->product_addon_group_id = $newAddOnGroup->id;
                    $newAddOnItem->save();
                }
            }

            // handle media
            foreach($adminVendorProduct->product->media as $media) {
                $newProduct->addMedia($media->getPath())->preservingOriginal()->toMediaCollection($media->collection_name);
            }
        }

    }

    public function categories()
    {
        return $this->belongsToMany(config('vtlabs_category.models.category'), 'ecommerce_vendor_categories');
    }

    public function productCategories()
    {
        return $this->belongsToMany(config('vtlabs_category.models.category'), 'ecommerce_vendor_product_categories')->distinct();
    }

    public function user()
    {
        return $this->belongsTo(config('auth.models.user'));
    }

    public function vendorProducts()
    {
        return $this->hasMany(VendorProduct::class);
    }

    public function availabilities()
    {
        return $this->hasMany(VendorAvailability::class, 'vendor_id');
    }

    public function isValidStatus(string $name, ?string $reason = null): bool
    {
        return in_array($name, config('vtlabs_ecommerce.vendor.status_list', []));
    }
}
