<?php

namespace Vtlabs\Ecommerce\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Depsimon\Wallet\Transaction;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Vtlabs\Ecommerce\Models\Order;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Ecommerce\Filters\OrderFilter;
use Vtlabs\Ecommerce\Models\DeliveryMode;
use Vtlabs\Ecommerce\Models\OrderDelivery;
use Vtlabs\Ecommerce\Services\OrderService;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ecommerce\Models\DeliveryProfile;
use Vtlabs\Ecommerce\Filters\DeliveryModeFilter;
use Vtlabs\Ecommerce\Models\DeliveryOrderRequest;
use Vtlabs\Ecommerce\Http\Resources\OrderResource;
use Vtlabs\Ecommerce\Events\UpdateDeliveryOrderRequest;
use Vtlabs\Ecommerce\Filters\DeliveryOrderRequestFilter;
use Vtlabs\Ecommerce\Http\Resources\DeliveryModeResource;
use Vtlabs\Ecommerce\Http\Resources\OrderDeliveryResource;
use Vtlabs\Ecommerce\Http\Resources\DeliveryProfileResource;
use Vtlabs\Ecommerce\Http\Resources\DeliveryOrderRequestResource;

class DeliveryProfileController extends Controller
{
    public function __construct()
    {
    }

    public function showMyProfile(Request $request)
    {
        $delivery = DeliveryProfile::findByUser(Auth::user()->id);
        return new DeliveryProfileResource($delivery);
    }

    public function update(DeliveryProfile $delivery, Request $request)
    {
        $request->validate([
            'meta' => 'sometimes|json',
            'is_online' => 'sometimes|boolean',
            'assigned' => 'sometimes|boolean',
            'longitude' => 'sometimes|numeric|min:-180|max:180',
            'latitude' => 'sometimes|numeric|min:-90|max:90'
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        $delivery->fill($request->only([
            'meta', 'is_online', 'longitude', 'latitude', 'assigned'
        ]));

        $delivery->save();

        return new DeliveryProfileResource($delivery->fresh());
    }

    public function summary(DeliveryProfile $delivery, Request $request)
    {
        $request->validate([
            'duration' => 'required|in:hours,days,months,years',
            'limit' => 'required|numeric'
        ]);

        $orders = Order::whereHas('delivery', function ($query) use ($delivery) {
            $query->where('delivery_profile_id', $delivery->id);
        });

        $ordersForCount = clone $orders;

        $ordersForDistance = clone $orders;

        $orders = OrderService::summarize($orders, $request->duration, $request->limit);

        $totalEarnings = Transaction::where('wallet_id', Auth::user()->wallet->id)->where('type', 'deposit')->whereDate('created_at', '<=', Carbon::now());

        if ($request->duration == 'hours') {
            $totalEarnings = $totalEarnings->whereDate('created_at', '>', Carbon::now()->subDays(1));
        }

        if ($request->duration == 'days') {
            $totalEarnings = $totalEarnings->whereDate('created_at', '>', Carbon::now()->subDays(7));
        }

        if ($request->duration == 'months') {
            $totalEarnings = $totalEarnings->whereDate('created_at', '>', Carbon::now()->subDays(30));
        }

        if ($request->duration == 'years') {
            $totalEarnings = $totalEarnings->whereDate('created_at', '>', Carbon::now()->subDays(365));
        }

        return response([
            "orders_count" => $ordersForCount->filter(['duration' => $request->duration], OrderFilter::class)->count(),
            "distance_travelled" => $ordersForDistance->filter(['duration' => $request->duration], OrderFilter::class)->sum('distance_travelled'),
            "earnings" => $totalEarnings->sum('amount'),
            'orders_chart_data' => $orders->get()
        ]);
    }

    public function currentOrder(DeliveryProfile $delivery, Request $request)
    {
        $orders = Order::filter(['delivery_profile' => $delivery->id], OrderFilter::class);

        $orders = $orders->otherCurrentStatus('complete');

        $order = $orders->ordered()->firstOrFail();

        return new OrderResource($order);
    }

    public function deliveryRequest(DeliveryProfile $delivery, Request $request)
    {
        $request->validate([
            'order' => 'sometimes|exists:ecommerce_orders,id',
            'status' => ['sometimes', Rule::in(config('vtlabs_ecommerce.order.status_list', []))]
        ]);

        request()->merge([
            "delivery" => $delivery->id,
            'not_assigned' => 1
        ]);

        $deliveryRequests = DeliveryOrderRequest::filter($request->all(), DeliveryOrderRequestFilter::class);

        $status = $request->status ?? 'pending';

        $deliveryRequests = $deliveryRequests->currentStatus([$status]);

        $deliveryRequest = $deliveryRequests->ordered()->firstOrFail();

        return new DeliveryOrderRequestResource($deliveryRequest);
    }

    public function updateDeliveryRequest(DeliveryOrderRequest $deliveryRequest, Request $request)
    {
        $request->validate([
            'status' => ['required', Rule::in(config('vtlabs_ecommerce.order.status_list', []))]
        ]);

         // if delivery user is trying to accept the request, first check again if order is already accepted by some other delivery user        
         if($request->status == 'accepted') {
            $order = Order::find($deliveryRequest->order->id); // fetch the latest instance of this order
            if($order->delivery != null) {
                return response()->json((object)[]);
            }
        }

        $deliveryRequest->setStatus($request->status);

        $deliveryRequest->save();

        event(new UpdateDeliveryOrderRequest($deliveryRequest));

        return new DeliveryOrderRequestResource($deliveryRequest);
    }

    public function deliveryModes(Request $request) 
    {
        $deliveryModes = DeliveryMode::filter($request->all(), DeliveryModeFilter::class);

        return DeliveryModeResource::collection($deliveryModes->paginate());
    }

    public function ratingStore(DeliveryProfile $delivery, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required'
        ]);

        $user = Auth::user();

        // $user->unrate($vendor);
        $user->rate($delivery, $request->rating, $request->review);

        $delivery->save();

        return response()->json([], 200);
    }

    public function ratingSummary(DeliveryProfile $delivery)
    {
        return response()->json([
            "average_rating" => $delivery->averageRating(User::class),
            "total_ratings" => $delivery->raters(User::class)->count(),
            "summary" => DB::table('ratings')->selectRaw('count(*) as total, ROUND(rating) as rounded_rating')
                ->where('rateable_type', DeliveryProfile::class)
                ->where('rateable_id', $delivery->id)
                ->where('rater_type', User::class)
                ->groupBy('rounded_rating')
                ->get()
        ]);
    }
}
