<?php

namespace Vtlabs\Ecommerce\Filters;

use Carbon\Carbon;
use EloquentFilter\ModelFilter;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use Vtlabs\Core\Helpers\CoreHelper;

class VendorFilter extends ModelFilter
{
    public function setup()
    {

        // filter blank profiles and non verified vendors
        if (!CoreHelper::isAdmin()) {
            $this->whereNotNull('name')->where('is_verified', true);
        }

        // filter vendors based on user location
        $distanceLimit = CoreHelper::settingsAsDictionary()['delivery_distance'] ?? 8000;

        if ($this->input('long') && $this->input('lat')) {
            $subqueryDistance = "ST_Distance_Sphere(Point(longitude,"
                . " latitude),"
                . " Point(" . $this->input('long') . ", " . $this->input('lat') . "))"
                . " as distance";

            $subqueryDistanceWhere = "ST_Distance_Sphere(Point(longitude,"
                . " latitude),"
                . " Point(" . $this->input('long') . ", " . $this->input('lat') . "))"
                . " < " . $distanceLimit;

            $this->select('*', DB::raw($subqueryDistance))
                ->whereRaw($subqueryDistanceWhere);
        }
    }

    public function search($search)
    {
        return CoreHelper::searchTranslated($this, 'name', App::getLocale(), $search);
    }

    public function name($name)
    {
        return CoreHelper::searchTranslated($this, 'name', App::getLocale(), $name);
    }

    public function category($id)
    {
        return $this->whereHas('categories', function ($query) use ($id) {
            return $query->where('id', $id)->orWhere('slug', $id);
        });
    }

    public function distance()
    {
    }

    public function discounted()
    {
        return $this->whereHas('vendorProducts', function ($query) {
            return $query->where('sale_price', '<>', 0);
        });
    }

    public function user($id)
    {
        return $this->whereHas('user', function ($query) use ($id) {
            return $query->where('name', 'like', '%' . $id . '%')
                ->orWhere('mobile_number', 'like', '%' . $id . '%')
                ->orWhere('email', 'like', '%' . $id . '%')
                ->orWhere('id', $id);
        });
    }

    public function meta($meta)
    {
        foreach ($meta as $key => $value) {
            return $this->where('meta->' . $key, $value);
        }
    }

    public function isOpen($dateTime)
    {
        return $this->whereHas('availabilities', function ($query) use ($dateTime) {
            $dayName = strtolower(Carbon::parse($dateTime)->shortEnglishDayOfWeek);
            $time = Carbon::parse($dateTime)->format('H:i');
            return $query->where('days', $dayName)->whereTime('from', '<=', $time)->whereTime('to', '>=', $time);
        });
    }

    public function sort($sort)
    {
        if ($sort == 'new') {
            return $this->where('plan_sort_order', '>=', 2)->orderBy('created_at', 'desc');
        }

        if ($sort == 'popular') {
            return $this->where('plan_sort_order', '>=', 2)->orderBy('orders_count', 'desc');
        }

        if ($sort == 'ratings') {
            return $this->where('plan_sort_order', '>=', 2)->orderBy('average_ratings', 'desc');
        }

        if ($sort == 'plan') {
            if ($this->input('search') || $this->input('category')) {
                return $this->orderByDesc('plan_sort_order');
            } else {
                return $this->where('plan_sort_order', 3)->orderByDesc('plan_sort_order');
            }
        }
    }
}
