<?php

namespace Vtlabs\Ecommerce\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Ecommerce\Models\Vendor;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Ecommerce\Models\Product;
use Vtlabs\Ecommerce\Models\VendorProduct;
use Vtlabs\Ecommerce\Filters\ProductFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ecommerce\Http\Resources\Admin\ProductAdminResource;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'title' => 'sometimes|string',
            'category' => 'sometimes|exists:categories,id'
        ]);

        $products = Product::filter($request->all(), ProductFilter::class);

        return ProductAdminResource::collection($products->paginate());
    }

    public function show($id)
    {
        return new ProductAdminResource(Product::findOrFail($id));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'detail_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'price' => 'required|numeric',
            'sale_price' => 'sometimes|nullable|numeric',
            'sale_price_from' => 'sometimes|date|nullable',
            'sale_price_to' => 'sometimes|date|nullable',
            'stock_quantity' => 'sometimes|numeric',
            'stock_low_threshold' => 'sometimes|numeric',
            'vendor_id' => 'required|exists:ecommerce_vendors,id',
            'categories' => 'sometimes|array|exists:categories,id',
            'addon_groups' => 'sometimes|array',
            'addon_groups.*.title_translations' => 'required|translation',
            'addon_groups.*.min_choices' => 'required|integer',
            'addon_groups.*.max_choices' => 'required|integer',
            'addon_groups.*.choices' => 'array|required',
            'addon_groups.*.choices.*.title_translations' => 'required|translation',
            'addon_groups.*.choices.*.price' => 'required',
            'images.*' => 'sometimes|image',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "detail" => request()->input('_detail') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $product = Product::create($request->only([
            'title', 'detail', 'meta', 'price'
        ]));

        $product->save();

        $product->categories()->sync($request->categories);

        // attach product categories with vendor
        $vendor = Vendor::find($request->vendor_id);

        $vendorProductCategories = [];
        for($i=0; $i < count($request->categories); $i++) {
            $vendorProductCategories[$request->categories[$i]] = ['product_id' => $product->id];
        }
        $vendor->productCategories()->wherePivot('product_id', $product->id)->sync($vendorProductCategories);

        $vendorProductAttributes = array_merge($request->only([
            'vendor_id', 'price', 'sale_price', 'sale_price_from', 'sale_price_to',
            'stock_quantity', 'stock_low_threshold'
        ]), ["product_id" => $product->id]);

        // create vendor specific details of product
        VendorProduct::create($vendorProductAttributes);

        // images
        if (!empty($request->images)) {
            foreach ($request->images as $image) {
                $product->addMedia($image)->toMediaCollection("images");
            }
        }

        // addons
        if ($request->has('addon_groups')) {
            $product->attachAddons($request->all());
        }

        return new ProductAdminResource($product->fresh());
    }

    public function update(Product $product, Request $request)
    {
        $request->validate([
            'title_translations' => 'required|translation',
            'detail_translations' => 'required|translation',
            'meta' => 'sometimes|json|nullable',
            'price' => 'required|numeric',
            'sale_price' => 'sometimes|nullable|numeric',
            'sale_price_from' => 'sometimes|date|nullable',
            'sale_price_to' => 'sometimes|date|nullable',
            'stock_quantity' => 'sometimes|numeric',
            'stock_low_threshold' => 'sometimes|numeric',
            'vendor_id' => 'required|exists:ecommerce_vendors,id',
            'categories' => 'sometimes|array|exists:categories,id',
            'addon_groups' => 'sometimes|array',
            'addon_groups.*.title_translations' => 'required|translation',
            'addon_groups.*.min_choices' => 'required|integer',
            'addon_groups.*.max_choices' => 'required|integer',
            'addon_groups.*.choices' => 'array|required',
            'addon_groups.*.choices.*.title_translations' => 'required|translation',
            'addon_groups.*.choices.*.price' => 'required',
            'images.*' => 'sometimes|image',
        ]);

        request()->merge([
            "title" => request()->input('_title'),
            "detail" => request()->input('_detail') ?? null
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta)
            ]);
        }

        $product->fill($request->only([
            'title', 'detail', 'meta', 'price'
        ]));

        $product->categories()->sync($request->categories);

        $product->save();

        // attach product categories with vendor
        $vendor = Vendor::find($request->vendor_id);
        $vendorProductCategories = [];
        for($i=0; $i < count($request->categories); $i++) {
            $vendorProductCategories[$request->categories[$i]] = ['product_id' => $product->id];
        }
        $vendor->productCategories()->wherePivot('product_id', $product->id)->sync($vendorProductCategories);

        // update vendor specific details of product
        $vendorProduct = VendorProduct::where('product_id', $product->id)->where('vendor_id', $request->vendor_id)->first();
        $vendorProduct->fill($request->only([
            'price', 'sale_price', 'sale_price_from', 'sale_price_to',
            'stock_quantity', 'stock_low_threshold'
        ]));
        $vendorProduct->save();

        // images
        if (!empty($request->images)) {
            $product->clearMediaCollection('images');
            foreach ($request->images as $image) {
                $product->addMedia($image)->toMediaCollection("images");
            }
        }

        // addons
        if ($request->has('addon_groups')) {
            $product->attachAddons($request->all());
        }

        return new ProductAdminResource($product->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        Product::findOrFail($id)->delete();

        return response()->json([], 200);
    }
}
