<?php

namespace Vtlabs\Ecommerce\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Illuminate\Database\Eloquent\Builder;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Ecommerce\Models\DeliveryProfile;
use Vtlabs\Ecommerce\Models\DeliveryPanicLog;
use Vtlabs\Ecommerce\Models\DeliveryOnlineLog;
use Vtlabs\Ecommerce\Filters\DeliveryProfileFilter;
use Vtlabs\Ecommerce\Http\Resources\DeliveryPanicLogResource;
use Vtlabs\Ecommerce\Http\Resources\DeliveryOnlineLogResource;
use Vtlabs\Ecommerce\Http\Resources\Admin\DeliveryProfileAdminResource;

class DeliveryProfileController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'user' => 'sometimes'
        ]);

        $deliveryProfiles = DeliveryProfile::filter($request->all(), DeliveryProfileFilter::class);

        return DeliveryProfileAdminResource::collection($deliveryProfiles->paginate());
    }

    public function show($id)
    {
        return new DeliveryProfileAdminResource(DeliveryProfile::findOrFail($id));
    }

    public function store(Request $request)
    {
        $request->validate([
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'is_online' => 'required|boolean',
            'assigned' => 'required|boolean',
            'longitude' => 'required|numeric|min:-180|max:180',
            'latitude' => 'required|numeric|min:-90|max:90'
        ]);

        // create user for delivery first  
        $userData = collect($request->only(['email', 'mobile_number', 'password']));
        $userData = $userData->merge(['name' => $userData['mobile_number'], 'mobile_verified' => 1, 'roles' => ['delivery']]);
        $user = UserService::create($userData);

        $deliveryProfile = DeliveryProfile::where('user_id', $user->id)->firstOrFail();

        $deliveryProfile->fill($request->only([
            'is_online', 'longitude', 'latitude', 'assigned'
        ]));

        $deliveryProfile->save();

        return new DeliveryProfileAdminResource($deliveryProfile->fresh());
    }

    public function update(Request $request, $id)
    {
        $deliveryProfile = DeliveryProfile::findOrFail($id);

        $request->validate([
            'is_online' => 'required|boolean',
            'assigned' => 'required|boolean',
            'longitude' => 'required|numeric|min:-180|max:180',
            'latitude' => 'required|numeric|min:-90|max:90'
        ]);

        $deliveryProfile->fill($request->only([
            'is_online', 'longitude', 'latitude', 'assigned'
        ]));

        $deliveryProfile->save();

        return new DeliveryProfileAdminResource($deliveryProfile->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        DeliveryProfile::findOrFail($id)->delete();

        return response()->json([], 200);
    }

    public function onlineLogs(DeliveryProfile $deliveryProfile, Request $request)
    {
        $deliveryOnlineLogs = DeliveryOnlineLog::where('delivery_profile_id', $deliveryProfile->id)->latest();
        return DeliveryOnlineLogResource::collection($deliveryOnlineLogs->paginate());
    }

    public function panicLogs(Request $request)
    {
        $deliveryPanicLogs = DeliveryPanicLog::latest();
        return DeliveryPanicLogResource::collection($deliveryPanicLogs->paginate());
    }
}
