<?php

namespace Vtlabs\Ecommerce\Services;

use Carbon\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Vtlabs\Ecommerce\Models\Order;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Ecommerce\Models\DeliveryProfile;
use Vtlabs\Ecommerce\Jobs\FirebaseUpdateOrder;
use Vtlabs\Ecommerce\Jobs\FirebaseUpdateDelivery;
use Vtlabs\Ecommerce\Models\DeliveryOrderRequest;
use Vtlabs\Ecommerce\Http\Resources\OrderResource;
use Vtlabs\Ecommerce\Filters\DeliveryProfileFilter;
use Vtlabs\Ecommerce\Http\Resources\DeliveryOrderRequestResource;

class OrderService
{
    private $order;

    public function __construct(Order $order)
    {
        $this->order = $order;
    }

    public function sendDeliveryRequest()
    {
        $deliveryAppEnabled = '1';
        if ($this->order->vendor->meta && isset($this->order->vendor->meta->delivery_app)) {
            $deliveryAppEnabled = $this->order->vendor->meta->delivery_app;
        }

        if ($deliveryAppEnabled != '1') {
            return;
        }

        $deliveryProfiles = $this->nearbyDeliveryProfiles()->get();

        foreach ($deliveryProfiles as $deliveryProfile) {
            $deliveryRequest = DeliveryOrderRequest::create([
                'delivery_profile_id' => $deliveryProfile->id,
                'order_id' => $this->order->id
            ]);

            $deliveryProfile->user->sendPushNotification(
                'delivery',
                __('vtlabs_ecommerce::messages.notification_new_order_delivery_title', ['id' => $this->order->id]),
                __('vtlabs_ecommerce::messages.notification_new_order_delivery_body')
            );

            config('queue.use_queue')
                ? FirebaseUpdateDelivery::dispatch($deliveryProfile, $deliveryRequest)
                : FirebaseUpdateDelivery::dispatchAfterResponse($deliveryProfile, $deliveryRequest);
        }
    }

    public function nearbyDeliveryProfiles()
    {
        $deliveryFilter = ['verified' => 1];
        switch ($this->order->order_type) {
            case Order::ORDER_TYPE_NORMAL:
                $deliveryFilter[] = ['vendor' => $this->order->vendor->id];
                break;
            case Order::ORDER_TYPE_CUSTOM:
                $deliveryFilter[] = ['address' => $this->order->sourceAddress->id];
                break;
        }

        $deliveryProfiles = DeliveryProfile::filter($deliveryFilter, DeliveryProfileFilter::class);

        // fiter out delivery profiles to whom request already sent
        $deliveryProfileIds = DeliveryOrderRequest::select('delivery_profile_id')
            ->where(function ($query) {
                // TODO: check for more conditions
                $query->where('order_id', $this->order->id);
            })->get()
            ->pluck('delivery_profile_id');
        $deliveryProfiles = $deliveryProfiles->whereNotIn('id', $deliveryProfileIds);

        return $deliveryProfiles;
    }

    public function notifyFirebase()
    {
        config('queue.use_queue')
            ? FirebaseUpdateOrder::dispatch($this->order->id)
            : FirebaseUpdateOrder::dispatchAfterResponse($this->order->id);
    }

    public static function summarize($query, $duration, $limit)
    {
        if ($duration == 'hours') {
            $query = $query->select(DB::raw('HOUR(created_at) as period'), DB::raw('count(*) as total'))
                ->whereDate('created_at', '>', Carbon::now()->subdays(1))
                ->whereDate('created_at', '<=', Carbon::now())
                ->groupBy(DB::raw('HOUR(created_at)'));
        }

        if ($duration == 'days') {
            $query = $query->select(DB::raw('DATE(created_at) as period'), DB::raw('count(*) as total'))
                ->whereDate('created_at', '>', Carbon::now()->subDays($limit))
                ->whereDate('created_at', '<=', Carbon::now())
                ->groupBy(DB::raw('DATE(created_at)'));
        }

        if ($duration == 'months') {
            $query = $query->select(DB::raw('MONTH(created_at) as period'), DB::raw('count(*) as total'))
                ->whereDate('created_at', '>', Carbon::now()->subDays($limit * 30))
                ->whereDate('created_at', '<=', Carbon::now())
                ->groupBy(DB::raw('MONTH(created_at)'));
        }

        if ($duration == 'years') {
            $query = $query->select(DB::raw('YEAR(created_at) as period'), DB::raw('count(*) as total'))
                ->whereDate('created_at', '>', Carbon::now()->subDays($limit * 365))
                ->whereDate('created_at', '<=', Carbon::now())
                ->groupBy(DB::raw('YEAR(created_at)'));
        }

        return $query;
    }

    public static function calculateReward($total)
    {   
        $settings = CoreHelper::settingsAsDictionary();

        return OrderService::getRewardFromSettings($settings, 1 , $total);
    }

    public static function getRewardFromSettings($settings, $optionCount, $total)
    {
        $key = 'card_rewards_option_' . $optionCount;
        $valueKey = 'card_rewards_percent_' . $optionCount;

        $rewardValue = null;

        if(Str::contains($settings[$key], '-')) {
            $range = explode('-', $settings[$key]);
            if($total >= $range[0] && $total <= $range[1]){
                $rewardValue = $settings[$valueKey];
            }
        } else if(Str::contains($settings[$key], '+')) {
            $val = explode('+', $settings[$key])[1];
            if($total >= floatval($val)) {
                $rewardValue = $settings[$valueKey];
            }
            
        }

        if($rewardValue != null || $optionCount >= 4) {
            return floatval($rewardValue);
        }

        return OrderService::getRewardFromSettings($settings, $optionCount+1, $total);
    }
}
