<?php

namespace Vtlabs\Core\Services;

use Illuminate\Support\Facades\Log;

class GoogleGeocodingService
{
    private $apiKey;
    private const GEOCODING_API_URL = 'https://maps.googleapis.com/maps/api/geocode/json';

    /**
     * Constructor for GoogleGeocodingService.
     *
     * @param string $apiKey Your Google Geocoding API key.
     */
    public function __construct(string $apiKey)
    {
        $this->apiKey = $apiKey;
    }

    /**
     * Gets the postal code (pincode) from given latitude and longitude.
     *
     * @param float $latitude The latitude coordinate.
     * @param float $longitude The longitude coordinate.
     * @return string|null The postal code if found, otherwise null.
     */
    public function getPostalCodeFromLatLng(float $latitude, float $longitude): ?string
    {
        $latlng = $latitude . ',' . $longitude;
        $url = self::GEOCODING_API_URL . '?latlng=' . urlencode($latlng) . '&key=' . urlencode($this->apiKey);

        // Optional: Add result_type to prioritize postal_code results
        // This can sometimes help narrow down the most relevant address component.
        // $url .= '&result_type=postal_code';

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true); // Return the response as a string
        curl_setopt($curl, CURLOPT_TIMEOUT, 10); // Set a timeout for the request

        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $error = curl_error($curl);
        curl_close($curl);

        if ($response === false) {
            Log::error("Google Geocoding API cURL Error: " . $error);
            return null;
        }

        if ($httpCode !== 200) {
            log::error("Google Geocoding API HTTP Error: " . $httpCode . " Response: " . $response);
            return null;
        }

        $data = json_decode($response, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            log::error("Google Geocoding API JSON Decode Error: " . json_last_error_msg());
            return null;
        }

        if (!isset($data['status']) || $data['status'] !== 'OK') {
            log::error("Google Geocoding API Status Error: " . ($data['status'] ?? 'Unknown') . " - " . ($data['error_message'] ?? 'No error message provided'));
            return null;
        }

        // Iterate through results to find the postal code
        if (isset($data['results']) && is_array($data['results']) && !empty($data['results'])) {
            foreach ($data['results'] as $result) {
                if (isset($result['address_components']) && is_array($result['address_components'])) {
                    foreach ($result['address_components'] as $component) {
                        // Look for a component with type 'postal_code'
                        if (isset($component['types']) && in_array('postal_code', $component['types'])) {
                            return $component['long_name']; // 'long_name' is the full postal code
                        }
                    }
                }
            }
        }

        return null; // Postal code not found in the response
    }
}
