<?php

namespace Vtlabs\Core\Http\Controllers\Api;

use Firebase\JWT\JWT;
use Vtlabs\Core\Events\LoggedIn;
use Vtlabs\Core\Models\User\User;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Vtlabs\Core\Services\FirebaseService;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Core\Http\Resources\UserResource;
use Vtlabs\Core\Http\Requests\Api\LoginRequest;
use Vtlabs\Core\Http\Requests\Api\CheckUserRequest;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * @group  Auth
 *
 * APIs for authentication
 */
class LoginController extends Controller
{
    /**
     * Login
     *
     * @bodyParam  token string required Firebase token. Example: firebase_token
     * @bodyParam  role string required Role of the user logging in. Example: customer
     * 
     */
    public function login(LoginRequest $request)
    {
        try {
            if ($this->_isSimulateUser()) {
                return $this->_prepareResponse(User::findOrFail(request()->input('simulate_user')));
            }

            if ($request->email) {
                // attempt email based authentication
                if (Auth::attempt(['email' => $request->email, 'password' => $request->password])) {
                    $user = Auth::user();
                } else {
                    throw new \Exception('The provided credentials do not match our records');
                }
            } else {
                // attempt token based authentication
                $payload = FirebaseService::getPayloadFromToken($request->token);

                // if ($decoded->iss !== FirebaseService::getFirebaseIss()) {
                //     throw new \Exception('Wrong FIREBASE_ISS provided');
                // }

                $mobile_number = array_key_exists('phone_number', $payload) ? $payload['phone_number'] : null;

                if (empty($mobile_number)) {
                    throw new \Exception('Mobile number not present in token');
                }

                $user = config('auth.models.user')::where('mobile_number', '=', $mobile_number)->firstOrFail();
            }

            if (!$user->hasRole($request->role)) {
                return response()->json(["message" => "Role mismatch"], 400);
            }

            // check if admin has blocked this user
            if (User::blockedByAdmin($user)) {
                return response()->json(["message" => "Blocked by admin"], 400);
            }

            // raise loggedin event
            event(new LoggedIn($user, $request->role));

            return $this->_prepareResponse($user);
        } catch (\Exception $ex) {
            throw new BadRequestHttpException($ex->getMessage());
        }
    }

    /**
     * Check if user exist
     * 
     * @bodyParam  mobile_number string required Mobile Number to check. Example +912324252627
     * 
     */
    public function checkUser(Request $request)
    {
        $request->validate(
            [
                'mobile_number' => 'required_without:email|exists:users,mobile_number',
                'email' => 'required_without:mobile_number|email|exists:users,email',
                'role' => 'required',
            ]
        );

        $user = User::query();

        if ($request->mobile_number) {
            $user = $user->where('mobile_number', $request->mobile_number);
        } elseif ($request->email) {
            $user = $user->where('email', $request->email);
        }

        $user = $user->firstOrFail();

        if (User::blockedByAdmin($user)) {
            return response()->json(["message" => "Blocked by admin"], 403);
        }

        if ($user->hasRole($request->role)) {
            return response()->json(["message" => "User exists"]);
        }

        return response()->json(["message" => "Role mismatch"], 400);
    }

    private function _prepareResponse($user)
    {
        $user->mobile_verified = 1;
        $user->save();
        $token = $user->createToken('Default')->accessToken;
        return response()->json(["token" => $token, "user" => new UserResource($user->refresh())]);
    }

    private function _isSimulateUser()
    {
        return request()->has('simulate_user') && request()->input('app_key') == env('APP_KEY');
    }
}
