<?php

namespace Vtlabs\Core\Http\Controllers\Api;

use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

/**
 * @group  User Management
 */
class GooglePlacesController extends Controller
{
    /**
     * The base URL for the Google Places API.
     * @var string
     */
    protected $baseUrl = 'https://maps.googleapis.com/maps/api/place/';

    /**
     * The Google Places API Key.
     * @var string
     */
    protected $apiKey;

    /**
     * Constructor to initialize the API key from environment variables.
     */
    public function __construct()
    {
        // Ensure the API key is set in your .env file
        // e.g., GOOGLE_PLACES_API_KEY=YOUR_API_KEY_HERE
        $this->apiKey = env('GOOGLE_PLACES_API_KEY');

        if (empty($this->apiKey)) {
            Log::error('Google Places API key is not set in the .env file.');
            // You might want to throw an exception or handle this more gracefully
            // depending on your application's error handling strategy.
        }
    }

    /**
     * Proxies requests to the Google Places Autocomplete API.
     *
     * @param Request $request The incoming HTTP request from your Flutter app.
     * @return \Illuminate\Http\JsonResponse
     */
    public function autocomplete(Request $request)
    {
        // Validate required parameters for autocomplete
        $request->validate([
            'input' => 'required|string',
            // Add other parameters if you need them, e.g., 'sessiontoken', 'types', 'components'
        ]);

        // Build the query parameters for the Google Places API
        $queryParams = array_merge($request->all(), [
            'key' => $this->apiKey,
        ]);

        try {
            // Make the request to Google Places Autocomplete API
            $response = Http::get($this->baseUrl . 'autocomplete/json', $queryParams);

            // Check if the request was successful (HTTP status 2xx)
            if ($response->successful()) {
                return response()->json($response->json());
            } else {
                // Log the error and return a server error response
                Log::error('Google Places Autocomplete API error:', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'query_params' => $queryParams
                ]);
                return response()->json([
                    'error' => 'Failed to fetch autocomplete predictions from Google Places API.',
                    'details' => $response->json() // Include Google's error details if available
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Catch any network or unexpected errors
            Log::error('Exception during Google Places Autocomplete API call: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'query_params' => $queryParams
            ]);
            return response()->json([
                'error' => 'An unexpected error occurred while processing your request.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function details(Request $request)
    {
        // Validate required parameters for place details
        $request->validate([
            'placeid' => 'required|string'
        ]);

        // Build the query parameters for the Google Places API
        $queryParams = array_merge($request->all(), [
            'key' => $this->apiKey,
        ]);

        try {
            // Make the request to Google Places Details API
            $response = Http::get($this->baseUrl . 'details/json', $queryParams);

            // Check if the request was successful (HTTP status 2xx)
            if ($response->successful()) {
                return response()->json($response->json());
            } else {
                // Log the error and return a server error response
                Log::error('Google Places Details API error:', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'query_params' => $queryParams
                ]);
                return response()->json([
                    'error' => 'Failed to fetch place details from Google Places API.',
                    'details' => $response->json() // Include Google's error details if available
                ], $response->status());
            }
        } catch (\Exception $e) {
            // Catch any network or unexpected errors
            Log::error('Exception during Google Places Details API call: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'query_params' => $queryParams
            ]);
            return response()->json([
                'error' => 'An unexpected error occurred while processing your request.',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}