<?php

namespace Vtlabs\Core\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Vtlabs\Report\Models\Report;
use Vtlabs\Core\Models\User\User;
use Spatie\Permission\Models\Role;
use Vtlabs\Core\Filters\UserFilter;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Maatwebsite\Excel\Facades\Excel;
use Rennokki\Plans\Models\PlanModel;
use Vtlabs\Core\Events\RoleAssigned;
use Vtlabs\Core\Exports\UsersExport;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Core\Http\Resources\BlockResource;
use Vtlabs\Core\Helpers\PushNotificationHelper;
use Vtlabs\Core\Http\Resources\Admin\UserAdminResource;
use Vtlabs\Core\Http\Resources\Admin\UserDetailAdminResource;
use Vtlabs\Core\Http\Resources\Admin\BlockedUserAdminResource;
use Vtlabs\Core\Http\Resources\Admin\ReportedUserAdminResource;

class UserController extends Controller
{
    private $userModel;
    public function __construct()
    {
        $this->userModel = config('auth.models.user');
    }

    public function index(Request $request)
    {
        $request->validate([
            'pagination' => 'sometimes|boolean',
            'email' => 'sometimes',
            'name' => 'sometimes',
            'mobile_number' => 'sometimes',
            'roles' => 'sometimes'
        ]);

        if (!Auth::user()->hasRole('superadmin') && !Auth::user()->hasRole('admin')) {
            request()->merge(['id' => Auth::id()]);
        }

        $users = $this->userModel::filter($request->all(), UserFilter::class)->oldest();

        // do not show superadmin users to admin
        if (Auth::user()->hasRole('admin')) {
            $users = $users->whereHas('roles', function ($query) {
                return $query->where('name', '!=', 'superadmin');
            });
        }

        if ($request->pagination == '0') {
            $users = $users->get();
        } else {
            $users = $users->paginate();
        }

        return UserAdminResource::collection($users);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|max:255',
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'mobile_verified' => 'required|boolean',
            'is_verified' => 'sometimes|boolean',
            'username' => 'sometimes|max:255',
            'image' => 'sometimes|image',
            'language' => 'sometimes|locale',
            'roles' => ['required', 'array', 'role'],
            'meta' => 'sometimes|json|nullable',
            'balance' => 'sometimes|numeric',
            'plan_id' => 'sometimes|exists:plans,id',
            'sub_users_count' => 'sometimes|integer',
            'categories' => 'sometimes|array|exists:categories,id',
            'postal_codes' => 'sometimes',
            'is_city_admin' => 'required|boolean',
        ]);

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        request()->merge([
            "password" => bcrypt($request->password),
            "postal_codes" => $request->postal_codes . ','
        ]);

        $user = UserService::create(collect($request->only(['name', 'email', 'mobile_number', 'password', 'mobile_verified', 'language', 'roles', 'username', 'meta', 'is_verified', 'postal_codes', 'is_city_admin'])));

        $user->fill($request->only(['sub_users_count']));
        if (!in_array('superadmin', $request->roles)) {
            $parentId = null;
            // if request contains parent_id then we priortize it
            if ($request->parent_id) {
                $parentId = $request->parent_id;
            } else {
                // otherwise we set the current user as parent
                $parentId = Auth::id();
            }
            $user->parent_id = $parentId;
        }
        $user->save();

        if ($request->image) {
            $user->addMedia($request->image)->toMediaCollection("images");
        }

        if ($request->categories) {
            $user->categories()->sync($request->categories);
        }

        // update user wallet
        if ($request->balance) {
            if ($request->balance > $user->balance) {
                // if new balance is greater, deposit amount in user wallet
                $user->deposit($request->balance - $user->balance);
            } else if ($request->balance < $user->balance) {
                // if new balance is lesser, withdraw amount from user wallet
                $user->withdraw($user->balance - $request->balance);
            }
        }

        // plan assignment
        if ($request->plan_id) {
            $plan = PlanModel::find($request->plan_id);
            if ($user->hasActiveSubscription()) {
                $user->cancelCurrentSubscription();
            }
            $user->subscribeTo($plan, $plan->duration);
        }

        return new UserAdminResource($user->fresh());
    }

    public function show(User $user)
    {
        return new UserDetailAdminResource($user);
    }

    public function update(Request $request, $id)
    {
        $user = $this->userModel::findOrFail($id);

        $request->validate([
            'name' => 'required|max:255',
            'email' => ['required', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'mobile_number' => ['required', Rule::unique('users')->ignore($user->id)],
            'mobile_verified' => 'required|boolean',
            'is_verified' => 'sometimes|boolean',
            'password' => 'sometimes|min:6',
            'username' => 'sometimes|max:255',
            'image' => 'sometimes|image',
            'language' => 'sometimes|locale',
            'roles' => ['required', 'array', 'role'],
            'meta' => 'sometimes|json|nullable',
            'balance' => 'sometimes|numeric',
            'sub_users_count' => 'sometimes|integer',
            'categories' => 'sometimes|array|exists:categories,id',
            'postal_codes' => 'sometimes',
            'is_city_admin' => 'required|boolean',
        ]);

        if ($request->has('password')) {
            request()->merge([
                "password" => bcrypt($request->password)
            ]);
        }

        if ($request->meta) {
            request()->merge([
                "meta" => json_decode($request->meta, true)
            ]);
        }

        if ($request->postal_codes && !str_ends_with($request->postal_codes, ",")) {
            request()->merge(["postal_codes" => $request->postal_codes . ',']);
        }

        $user->fill($request->only(['name', 'email', 'mobile_number', 'password', 'mobile_verified', 'language', 'username', 'meta', 'is_verified', 'sub_users_count', 'postal_codes', 'is_city_admin']));
        $user->save();

        if ($request->image) {
            $user->clearMediaCollection('images');
            $user->addMedia($request->image)->toMediaCollection("images");
        }

        // attach roles
        $user->syncRoles($request->roles);

        event(new RoleAssigned($user, $request->roles));

        if ($request->categories) {
            $user->categories()->sync($request->categories);
        } else {
            $user->categories()->sync([]);
        }

        // update user wallet
        if ($request->balance) {
            if ($request->balance > $user->balance) {
                // if new balance is greater, deposit amount in user wallet
                $user->deposit($request->balance - $user->balance);
            } else if ($request->balance < $user->balance) {
                // if new balance is lesser, withdraw amount from user wallet
                $user->withdraw($user->balance - $request->balance);
            }
        }

        // plan assignment
        if ($request->plan_id) {
            $plan = PlanModel::find($request->plan_id);
            if ($user->hasActiveSubscription()) {
                if ($request->plan_id != $user->activeSubscription()->plan->id) {
                    $user->cancelCurrentSubscription();
                    $user->subscribeTo($plan, $plan->duration);
                }
            } else {
                $user->subscribeTo($plan, $plan->duration);
            }
        } else {
            $user->cancelCurrentSubscription();
        }

        return new UserAdminResource($user->fresh());
    }

    public function destroy($id)
    {
        Gate::authorize('delete');

        $this->userModel::findOrFail($id)->delete();

        return response()->json([], 200);
    }

    public function roles()
    {
        return response()->json(Role::all());
    }

    public function reports()
    {
        $reports = Report::where('reportable_type', User::class)->orderByDesc('created_at')->paginate();
        return ReportedUserAdminResource::collection($reports);
    }

    public function deleteReport(Report $report)
    {
        $report->delete();
        return response()->json([], 200);
    }

    public function export()
    {
        return Excel::download(new UsersExport, 'users.xlsx');
    }

    public function sendPushNotificationToAll(Request $request)
    {
        $request->validate([
            'role' => 'required|role',
            'message' => 'required|max:255'
        ]);

        PushNotificationHelper::sendNotificationToAll($request->roles, $request->message);

        return response()->json([], 201);
    }

    public function sendNotificationToUser(User $user, Request $request)
    {
        $request->validate([
            'message' => 'required'
        ]);

        $result = $user->sendPushNotification('customer', $request->message, null, []);

        if (!$result) {
            return response()->json(['meesage' => 'Unable to send notification'], 400);
        }

        return response()->json([], 200);
    }

    public function sendNotificationToAll(Request $request)
    {
        $request->validate([
            'message' => 'required'
        ]);

        User::sendPushNotificationToAll('customer', $request->message, null, []);

        return response()->json([], 200);
    }

    public function blockList(Request $request)
    {
        return BlockedUserAdminResource::collection(Auth::user()->blockedList(User::class)->paginate());
    }


    public function block(User $user, Request $request)
    {
        // if superadmin blocks the user then we will delete all the user's existing tokens and will not let him login on the platform

        $blocker = Auth::user();
        $blocked = false;

        if ($blocker->hasRole('superadmin')) {

            if ($blocker->id != $user->id) {
                if ($blocker->hasBlocked($user)) {
                    // unblock user if already blocked
                    $blocker->unblock($user);
                } else {
                    $blocked = true;
                    $blocker->block($user, ['reason' => 'remove from platform']);

                    // delete all existing tokens of a user
                    $user->tokens()->delete();
                }
            }

            return response(["block" => $blocked], 200);
        }

        return response(["block" => false], 401);
    }
}
