<?php

namespace Vtlabs\Core\Http\Controllers\Api;

use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Core\Http\Controllers\Controller;

/**
 * @group  Media
 *
 * APIs for file upload
 */
class StorageController extends Controller
{
    /**
     * Upload a file
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function upload(Request $request)
    {
        // Define allowed file types
        $allowedMimeTypes = [
            // Images
            'image/jpeg',
            'image/png',
            'image/gif',

            // Documents
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document', // .docx
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', // .xlsx
            'text/plain',

            // Videos
            'video/mp4',
            'video/mpeg',
            'video/quicktime',    // .mov
            'video/x-msvideo',    // .avi
            'video/x-ms-wmv',     // .wmv
            'video/x-flv',        // .flv
            'video/webm',         // .webm
            'video/3gpp',         // .3gp
            'video/x-matroska'    // .mkv
        ];

        // Enhanced validation
        $request->validate([
            'file' => [
                'required',
                'file',
                'max:102400', // 100MB max file size
                'mimes:jpeg,png,gif,pdf,doc,docx,xls,xlsx,txt,mp4,mpeg,mov,avi,wmv,flv,webm,3gp,mkv',
                function ($attribute, $value, $fail) use ($allowedMimeTypes) {
                    if (!in_array($value->getMimeType(), $allowedMimeTypes)) {
                        $fail('The file type is not allowed.');
                    }
                },
            ]
        ]);

        try {
            // Sanitize original filename
            $originalName = preg_replace('/[^a-zA-Z0-9.]/', '_', $request->file('file')->getClientOriginalName());

            // Get file extension
            $extension = $request->file('file')->getClientOriginalExtension();

            // Generate unique filename with sanitized original name
            $fileName = CoreHelper::generateRandomString(8) . '_' . time() . '.' . $extension;

            // Additional security check for file extension
            if (!in_array(strtolower($extension), ['jpeg', 'jpg', 'png', 'gif', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'mp4', 'mpeg', 'mov', 'avi', 'wmv', 'flv', 'webm', '3gp', 'mkv'])) {
                return response()->json(['message' => 'Invalid file extension'], 422);
            }

            // Store the file in the public disk with additional directory structure
            $path = $request->file('file')->storeAs('uploads/', $fileName, 'public');

            // Generate the full URL for the stored file
            $url = asset('storage/' . $path);

            return response()->json(['url' => $url]);
        } catch (\Exception $e) {
            Log::error('File upload failed: ' . $e->getMessage());

            return response()->json(['message' => 'File upload failed'], 500);
        }
    }
}
