<?php

namespace Vtlabs\Core\Helpers;

use EloquentFilter\ModelFilter;
use Vtlabs\Core\Models\Setting;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Auth;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\Storage;
use Firebase\JWT\Key;
use Spatie\MediaLibrary\HasMedia;

class CoreHelper
{
    public static function transformRequestForTranslation($key, $input)
    {
        $translations = [];
        foreach ($input as $value) {
            $translations[$value['language']] = $value['text'];
        }
        request()->merge(["_" . $key => $translations]);
        request()->merge([$key => $translations]);
    }

    public static function rolesForValidation()
    {
        $roles = Role::all()->pluck('name')->filter(function ($v, $k) {
            return $v[0] != '_';
        });
        return $roles->implode(',');
    }

    public static function isValidLocale($locale)
    {
        return in_array($locale, explode(",", config('app.locales')));
    }

    public static function tempPath(UploadedFile $file)
    {
        return Storage::path(Storage::putFile('temp', $file));
    }

    public static function getFirebaseKeyIds()
    {
        $publicKeyURL = 'https://www.googleapis.com/robot/v1/metadata/x509/securetoken@system.gserviceaccount.com';
        if (!ini_get('allow_url_fopen')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $publicKeyURL);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $publicKey = curl_exec($ch);
            curl_close($ch);
            $kids = json_decode($publicKey, true);
        } else {
            $kids = json_decode(file_get_contents($publicKeyURL), true);
        }
        return new Key($kids[array_key_first($kids)], 'RS256');
    }

    public static function settingsAsDictionary()
    {
        $settings = [];
        foreach (Setting::all() as $setting) {
            $settings[$setting->key] = $setting->value;
        }
        return $settings;
    }

    public static function fillTranslation($model, $fields)
    {
        $locales = explode(",", config('app.locales'));
        foreach ($fields as $key => $value) {
            // only update translation if it's value is changed
            if ($model->{$key} != $fields[$key]) {
                $translations = [];
                foreach ($locales as $locale) {
                    $translations[$locale] = $value;
                }
                $model->setTranslations($key, $translations);
            }
        }
    }

    public static function dumpQueryLog($query)
    {
        DB::connection()->enableQueryLog();
        $query->get();
        dd(DB::getQueryLog());
    }

    public static function searchTranslated($filter, string $columnName, string $locale, string $value)
    {
        return $filter->where(DB::raw('lower(' . $columnName . ')'), 'LIKE', '%' . strtolower($value) . '%');
    }

    public static function isAdmin()
    {
        return (Auth::user() && Auth::user()->hasRole('admin'));
    }

    public static function generatePin($digits = 6)
    {
        $i = 0;
        $pin = "";

        while ($i < $digits) {
            $pin .= mt_rand(0, 9);
            $i++;
        }

        return $pin;
    }

    public static function requestMergeJsonField($fields = ['meta'])
    {
        $request = request();

        foreach ($fields as $field) {
            if ($request->has($field)) {
                $request->merge([$field => $request->$field ? json_decode($request->$field, true) : null]);
            }
        }
    }

    public static function requestSaveImage($model, $requestField, $update = false, $collection = 'images')
    {
        $request = request();
        if ($request->has($requestField)) {
            if ($update) {
                $model->clearMediaCollection($collection);
            }
            $model->addMedia($request->$requestField)->toMediaCollection($collection);
        }
    }

    public static function fillableFieldsFromRequestRules($rules, $except = []): array
    {
        $fields =  array_filter(array_keys($rules), function ($v) use ($except) {
            return !str_contains($v, '.*') && array_search($v, $except) === false;
        });

        // convert _translations fields to name used in table
        $fields = array_map(function ($value): string {
            if (str_contains($value, '_translations')) {
                return str_replace('_translations', '', $value);
            }
            return $value;
        }, $fields);

        return $fields;
    }

    public static function generateRandomString(int $length = 8, $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'): string
    {
        $randomString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[random_int(0, strlen($characters) - 1)];
        }
        return $randomString;
    }
}
