<?php

namespace Vtlabs\Carrental\Http\Controllers\Api;

use Illuminate\Http\Request;
use Vtlabs\Carrental\Models\Car;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Carrental\Filters\CarFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carrental\Http\Resources\CarResource;
use Vtlabs\Carrental\Http\Resources\CarRatingResource;

class CarController extends Controller
{
    public function index(Request $request)
    {
        $cars = Car::filter($request->all(), CarFilter::class)->paginate();

        return CarResource::collection($cars);
    }

    public function show(Car $car)
    {
        return new CarResource($car);
    }

    public function store(Request $request)
    {
        $request->validate($this->rules());

        CoreHelper::requestMergeJsonField();

        $car = new Car();

        $car->fill($request->only(Corehelper::fillableFieldsFromRequestRules($this->rules(), ['name', 'details','features', 'image', 'addons', 'fee'])));

        CoreHelper::fillTranslation($car, $request->only(['name', 'detail']));

        $car->save();

        $car->features()->sync($request->features);

        if ($request->addons) {
            $car->addons()->createMany($request->addons);
        }

        if ($request->fee) {
            $car->fee()->createMany($request->fee);
        }

        if (!empty($request->image_url)) {
            $car->addMediaFromUrl($request->image_url)->toMediaCollection("images");
        }

        return new CarResource($car);
    }

    public function update(Request $request, Car $car)
    {
        $request->validate($this->rules());

        CoreHelper::requestMergeJsonField();

        $car->fill($request->only(Corehelper::fillableFieldsFromRequestRules($this->rules(), ['features', 'image', 'addons', 'fee'])));
        $car->save();

        $car->features()->sync($request->features);

        if ($request->addons) {
            $car->addons()->delete();
            $car->addons()->createMany($request->addons);
        }

        if ($request->fee) {
            $car->fee()->delete();
            $car->fee()->createMany($request->fee);
        }

        CoreHelper::requestSaveImage($car, 'image');

        return new CarResource($car);
    }

    public function destroy(Car $car)
    {
        Gate::authorize('delete');
        
        $car->delete();

        return response()->json([], 204);
    }

    public function ratingList(Car $car, Request $request)
    {
        return CarRatingResource::collection($car->raters(User::class)->orderByDesc('pivot_created_at')->paginate());
    }

    public function ratingStore(Car $car, Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric',
            'review' => 'required'
        ]);

        $user = Auth::user();

        $user->rate($car, $request->rating, $request->review);

        return response()->json([], 200);
    }

    public function ratingSummary(Car $car)
    {
        return response()->json([
            "average_rating" => $car->averageRating(User::class),
            "total_ratings" => $car->raters(User::class)->count(),
            "summary" => DB::table('ratings')->selectRaw('count(*) as total, ROUND(rating) as rounded_rating')
                ->where('rateable_type', CarProfile::class)
                ->where('rateable_id', $car->id)
                ->where('rater_type', User::class)
                ->groupBy('rounded_rating')
                ->get()
        ]);
    }

    private function rules(): array 
    {
        return [
            'name' => 'required',
            'details' => 'required',
            'meta' => 'sometimes|json|nullable',
            'image_url' => 'sometimes',
            'address' => 'required',
            'longitude' => 'required|numeric|min:-180|max:180',
            'latitude' => 'required|numeric|min:-90|max:90',
            'provider_id' => 'required|exists:carrental_providers,id',
            'brand_id' => 'required|exists:categories,id',
            'fueltype_id' => 'required|exists:categories,id',
            'type_id' => 'required|exists:categories,id',
            'features' => 'required|array|exists:categories,id',
            'addons.*' => 'sometimes|array',
            'addons.*.title' => 'required',
            'addons.*.fee' => 'required',
            'addons.*.is_per_day' => 'sometimes',
            'fee.*' => 'required|array',
            'fee.*.days' => 'required',
            'fee.*.fee' => 'required',
        ];
    }
}
