<?php

namespace Vtlabs\Carrental\Http\Controllers\Api\Admin;

use Illuminate\Support\Arr;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Carrental\Models\Provider;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Carrental\Filters\ProviderFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carrental\Http\Resources\Admin\ProviderAdminResource;

class ProviderController extends Controller
{
    public function index(Request $request)
    {
        if(!Auth::user()->hasRole('admin')) {
            request()->merge(['user_id' => Auth::id()]);
        }

        $providers = Provider::filter($request->all(), ProviderFilter::class)->latest();

        if($request->pagination == "0") {
            $providers = $providers->get();
        } else {
            $providers = $providers->paginate();
        }

        return ProviderAdminResource::collection($providers);
    }

    public function show(Provider $provider)
    {
        return new ProviderAdminResource($provider);
    }

    public function store(Request $request)
    {
        $request->validate($this->rules());

        // create a user for provider  
        request()->merge([
            "password" => bcrypt($request->password)
        ]);
        $userName = $request->name[array_keys(($request->name))[0]];
        $userData = collect($request->only(['email', 'mobile_number', 'password']));
        $userData = $userData->merge(['name' => $userName, 'mobile_verified' => 1, 'roles' => ['carprovider']]);
        $user = UserService::create($userData);

        $provider = Provider::findByUser($user->id);

        CoreHelper::requestMergeJsonField();

        $provider->fill($request->only(Corehelper::fillableFieldsFromRequestRules($this->rules(), ['email', 'mobile_number', 'password'])));
        $provider->save();

        return new ProviderAdminResource($provider);
    }

    public function update(Request $request, Provider $provider)
    {
        $request->validate(Arr::except($this->rules(), ['email', 'mobile_number', 'password']));

        CoreHelper::requestMergeJsonField();

        $provider->fill($request->only(Corehelper::fillableFieldsFromRequestRules($this->rules())));
        $provider->save();

        return new ProviderAdminResource($provider);
    }

    public function destroy(Provider $provider)
    {
        Gate::authorize('delete');
        
        $provider->delete();

        return response()->json([], 204);
    }

    private function rules(): array 
    {
        return [
            'email' => 'sometimes|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'name_translations' => 'required|json|translation',
            'details_translations' => 'required|json|translation',
            'meta' => 'sometimes|json|nullable'
        ];
    }
}
