<?php

namespace Vtlabs\Carrental\Http\Controllers\Api\Admin;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Carrental\Models\Car;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Appointment\Models\Appointment;
use Vtlabs\Core\Http\Controllers\Controller;

class DashboardController extends Controller
{
    public function bookingAnalytics(Request $request)
    {
        $request->validate([
            'period' => 'required|in:week,month,year,all'
        ]);

        $days = 7;
        $period = $request->period;
        $groupBy = "DATE('a.created_at')";
        $label = 'DATE(a.created_at) as created_at';

        if (!Auth::user()->hasRole('admin')) {
            $carIds = Car::whereHas('provider', function ($query) {
                return $query->Where('user_id', Auth::id());
            })->pluck('id')->toArray();
        }

        switch ($period) {
            case 'week':
                $days = 7;
                $groupBy = 'DATE(a.created_at)';
                $label = 'DATE(a.created_at) as created_at';
                break;
            case 'month':
                $days = 30;
                $groupBy = 'DATE(a.created_at)';
                $label = 'DATE(a.created_at) as created_at';
                break;
            case 'year':
                $days = 365;
                $groupBy = 'MONTH(a.created_at)';
                $label = 'MONTH(a.created_at) as created_at';
                break;
            case 'all':
                $days = 365 * 5;
                $groupBy = 'YEAR(a.created_at)';
                $label = 'YEAR(a.created_at) as created_at';
                break;
        }

        $bookingsChartData = DB::table('appointments as a')->select(DB::raw($label), DB::raw('count(*) as amount'))
            ->whereDate('a.created_at', '>', Carbon::now()->subDays($days))
            ->whereDate('a.created_at', '<=', Carbon::now())
            ->join('statuses as s', 'a.id', '=', 's.model_id')
            ->where('s.model_type', '=', Appointment::class)
            ->groupBy(DB::raw($groupBy));

        if($carIds) {
            $bookingsChartData = $bookingsChartData->whereIn('appointee_id', $carIds);
        }

        $completeBookingsChartData = clone $bookingsChartData;
        $completeBookingsChartData = $completeBookingsChartData->where('s.name', 'complete')->get()->mapWithKeys(function ($item) {
            return [$item->created_at => $item->amount];
        })->all();

        $otherBookingsChartData = clone $bookingsChartData;
        $bookingOtherChartData = $otherBookingsChartData->where('s.name', '<>', 'complete')->get()->mapWithKeys(function ($item) {
            return [$item->created_at => $item->amount];
        })->all();

        $bookingsChartData = $bookingsChartData->get();

        $totalCountQuery = Appointment::whereRaw("1=1");
        $lastMonthCountQuery = Appointment::whereDate('created_at', '>', Carbon::now()->subDays(30));
        $lastWeekCountQuery = Appointment::whereDate('created_at', '>', Carbon::now()->subDays(7));
        $todayCountQuery = Appointment::whereDate('created_at', '=', Carbon::now());
        if ($carIds) {
            $totalCountQuery = $totalCountQuery->whereIn('appointee_id', $carIds);
            $lastMonthCountQuery = $lastMonthCountQuery->whereIn('appointee_id', $carIds);
            $lastWeekCountQuery = $lastWeekCountQuery->whereIn('appointee_id', $carIds);
            $todayCountQuery = $todayCountQuery->whereIn('appointee_id', $carIds);
        }

        $summary = [
            ["title" => "Total", "value" => $totalCountQuery->count()],
            ["title" => "Last Month", "value" => $lastMonthCountQuery->count()],
            ["title" => "Last Week", "value" => $lastWeekCountQuery->count()],
            ["title" => "Today", "value" => $todayCountQuery->count()]
        ];

        $chartLabel = array_map(function ($date) use ($period) {
            if ($period == 'week') {
                $in = date_create($date);
                return date_format($in, "D");
            }
            if ($period == 'month') {
                $in = date_create($date);
                return date_format($in, "d");
            }
            if ($period == 'year') {
                $in = date_create('2013-10-27'); // create a sample date object
                $in->setDate($in->format('Y'), $date, $in->format('d'));
                return $in->format("M");
            }
            if ($period == 'all') {
                $in = date_create('2013-10-27'); // create a sample date object
                $in->setDate($date, $in->format('m'), $in->format('d'));
                return $in->format("Y");
            }
            return $date;
        }, $bookingsChartData->pluck('created_at')->toArray());

        $allBookingsCount = [];
        $completeBookingsCount = [];
        $otherBookingsCount = [];

        foreach ($bookingsChartData as $bookingData) {
            $allBookingsCount[]  = $bookingData->amount;
            $completeBookingsCount[] = $completeBookingsChartData[$bookingData->created_at] ?? 0;
            $otherBookingsCount[] = $bookingOtherChartData[$bookingData->created_at] ?? 0;
        }

        return response()->json([
            "chart" => [
                "chartLabel" => $chartLabel,
                "linesData" => [
                    $allBookingsCount,
                    $otherBookingsCount,
                    $completeBookingsCount,
                ]
            ],
            "summary" => $summary
        ]);
    }

    public function progressInfo(Request $request)
    {
        return response()->json([
            $this->revenueProgressInfo(),
            $this->bookingProgressInfo()
        ]);
    }

    private function revenueProgressInfo()
    {
        $today = Appointment::whereDate('created_at', Carbon::now())->sum('amount');
        $yesterday = Appointment::whereDate('created_at', Carbon::now()->subDays(1))->sum('amount');

        return $this->progressInfoFormat($today, $yesterday, "Revenue");
    }

    private function bookingProgressInfo()
    {
        $today = Appointment::whereDate('created_at', Carbon::now())->count();
        $yesterday = Appointment::whereDate('created_at', Carbon::now()->subDays(1))->count();

        return $this->progressInfoFormat($today, $yesterday, "Bookings");
    }

    private function progressInfoFormat($today, $yesterday, $title)
    {
        $percentChange =  $today > 0 ? (($yesterday - $today) / $today) * 100 : ($yesterday > 0 ? -100 : 0);
        $percentChange = round($percentChange, 2);

        $changeDescription = "";
        if ($percentChange > 0) {
            $changeDescription = "Better than yesterday ($percentChange%)";
        } else if ($percentChange == 0) {
            $changeDescription = "No change since yesterday";
        } else if ($percentChange < 0) {
            $changeDescription = "Lesser than yesterday ($percentChange%)";
        }

        return [
            "title" => "Today's $title",
            "value" => $today,
            "activeProgress" => $percentChange,
            "description" => $changeDescription
        ];
    }
}
