<?php

namespace Vtlabs\Carrental\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Vtlabs\Carrental\Models\Car;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Carrental\Filters\CarFilter;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carrental\Http\Resources\Admin\CarAdminResource;

class CarController extends Controller
{
    public function index(Request $request)
    {
        if(!Auth::user()->hasRole('admin')) {
            request()->merge(['user_id' => Auth::id()]);
        }
        
        $cars = Car::filter($request->all(), CarFilter::class)->latest()->paginate();

        return CarAdminResource::collection($cars);
    }

    public function show(Car $car)
    {
        return new CarAdminResource($car);
    }

    public function store(Request $request)
    {
        $request->validate($this->rules());

        CoreHelper::requestMergeJsonField();

        $car = Car::create($request->only(Corehelper::fillableFieldsFromRequestRules($this->rules(), ['features', 'image', 'addons', 'fee'])));

        $car->features()->sync($request->features);

        if ($request->addons) {
            $car->addons()->createMany($request->addons);
        }

        if ($request->fee) {
            $car->fee()->createMany($request->fee);
        }

        CoreHelper::requestSaveImage($car, 'image');

        return new CarAdminResource($car);
    }

    public function update(Request $request, Car $car)
    {
        $request->validate($this->rules());

        CoreHelper::requestMergeJsonField();

        $car->fill($request->only(Corehelper::fillableFieldsFromRequestRules($this->rules(), ['features', 'image', 'addons', 'fee'])));
        $car->save();

        $car->features()->sync($request->features);

        $car->addons()->delete();
        if ($request->addons) {
            $car->addons()->createMany($request->addons);
        }

        $car->schedule()->where('type', 'off')->delete();
        if ($request->schedule) {
            foreach($request->schedule as $schedule) {
                $car->schedule()->create([
                    'date_from' => $schedule['date_from'],
                    'date_to' => $schedule['date_from'],
                    'type' => 'off'
                ]);
            }            
        }

        CoreHelper::requestSaveImage($car, 'image');

        return new CarAdminResource($car);
    }

    public function destroy(Car $car)
    {
        Gate::authorize('delete');
        
        $car->delete();

        return response()->json([], 204);
    }

    public function schedule(Car $car)
    {
        return response()->json(DB::table('carrental_schedule')->whereDate('date_from', '>=', now())->get(), 200);
    }

    private function rules(): array 
    {
        return [
            'name_translations' => 'required|json|translation',
            'details_translations' => 'required|json|translation',
            'meta' => 'sometimes|json|nullable',
            'image' => 'sometimes|image',
            // 'fee' => 'required',
            'address' => 'required',
            'longitude' => 'required|numeric|min:-180|max:180',
            'latitude' => 'required|numeric|min:-90|max:90',
            'provider_id' => 'required|exists:carrental_providers,id',
            'brand_id' => 'required|exists:categories,id',
            'fueltype_id' => 'required|exists:categories,id',
            'type_id' => 'required|exists:categories,id',
            'features' => 'required|array|exists:categories,id',
            'addons.*' => 'sometimes|array',
            'addons.*.title' => 'required',
            'addons.*.fee' => 'required',
            'addons.*.is_per_day' => 'sometimes',
            'fee.*' => 'required|array',
            'fee.*.days' => 'required',
            'fee.*.fee' => 'required',
        ];
    }
}
