<?php

namespace Vtlabs\Carpool\Models;

use DateTime;
use Carbon\Carbon;
use Illuminate\Http\Request;
use EloquentFilter\Filterable;
use Vtlabs\Core\Models\Setting;
use Vtlabs\Core\Models\User\User;
use Vtlabs\Payment\Traits\CanPay;
use Illuminate\Support\Facades\DB;
use Rennokki\Plans\Traits\HasPlans;
use Rennokki\Rating\Traits\CanBeRated;
use Illuminate\Database\Eloquent\Model;
use Rennokki\Rating\Contracts\Rateable;
use Vtlabs\Carpool\Models\PickupLocation;
use Vtlabs\Core\Traits\CoreHasMediaTrait;
use Spatie\MediaLibrary\HasMedia\HasMedia;
use ChristianKuri\LaravelFavorite\Traits\Favoriteable;

class CarpoolProfile extends Model implements HasMedia, Rateable
{
    use Filterable,
        Favoriteable,
        CoreHasMediaTrait,
        CanBeRated,
        HasPlans,
        CanPay;


    protected $table = 'carpool_profiles';

    protected $guarded = [];

    protected $casts = [
        'meta' => 'json',
        'vehicle_details' => 'json',
        'is_verified' => 'integer',
        'seats' => 'integer',
        'price' => 'float'
    ];

    public static function findByUser($userId)
    {
        return CarpoolProfile::where('user_id', $userId)->firstOrFail();
    }

    public static function search($user, Request $request)
    {
        $distanceDelta = 1000;
        $minutesDelta = 60;

        $distanceSetting = Setting::where('key', 'distance_limit')->first();
        if ($distanceSetting) {
            $distanceDelta = $distanceSetting->value ? (int) $distanceSetting->value : $distanceDelta;
        }

        $minutesSetting = Setting::where('key', 'minutes_delta')->first();
        if ($minutesSetting) {
            $minutesDelta = $minutesSetting->value ? (int) $minutesSetting->value : $minutesDelta;
        }

        // show providers listed with in particular distance
        $subqueryDistanceFrom = "ST_Distance_Sphere(Point(pl1.longitude,"
            . " pl1.latitude),"
            . " Point($request->long_from, $request->lat_from ))"
            . " as distance_from";
        $subqueryDistanceTo = "ST_Distance_Sphere(Point(pl2.longitude,"
            . " pl2.latitude),"
            . " Point($request->long_to, $request->lat_to))"
            . " as distance_to";

        // calculate number of seats left
        $subqueryRides = "CAST(seats - COALESCE((SELECT SUM(seats) from carpool_rides where date(ride_on) = '" . Carbon::createFromFormat("Y-m-d", $request->date)->toDateString() . "'"
            . " and time(ride_on) <= '" . Carbon::createFromFormat("H:i", $request->time)->addMinutes($minutesDelta)->toTimeString() . "'"
            . " and time(ride_on) >= '" . Carbon::createFromFormat("H:i", $request->time)->subMinutes($minutesDelta)->toTimeString() . "'"
            // . " and status in ('pending', 'accepted')"
            . " and profile_id=carpool_profiles.id), 0) as SIGNED) as seats_available";

        $subqueryDistanceWhereFrom = "ST_Distance_Sphere(Point(pl1.longitude,"
            . " pl1.latitude),"
            . " Point($request->long_from, $request->lat_from ))"
            . " < " . $distanceDelta;

        $subqueryDistanceWhereTo = "ST_Distance_Sphere(Point(pl2.longitude,"
            . " pl2.latitude),"
            . " Point($request->long_to, $request->lat_to ))"
            . " < " . $distanceDelta;

        // check if driver has enough seats left
        $subqueryRidesWhere = "CAST(seats - COALESCE((SELECT SUM(seats) from carpool_rides where date(ride_on) = '" . Carbon::createFromFormat("Y-m-d", $request->date)->toDateString() . "'"
            . " and time(ride_on) <= '" . Carbon::createFromFormat("H:i", $request->time)->addMinutes($minutesDelta)->toTimeString() . "'"
            . " and time(ride_on) >= '" . Carbon::createFromFormat("H:i", $request->time)->subMinutes($minutesDelta)->toTimeString() . "'"
            // . " and status in ('pending', 'accepted')"
            . " and profile_id=carpool_profiles.id), 0) as SIGNED) >= " . $request->seats;

        $providers = CarpoolProfile::select(
            'carpool_profiles.*',
            'pl1.id as pickup_location_id',
            'pl2.id as drop_location_id',
            'pl1.address as address_from',
            'pl1.longitude as longitude_from',
            'pl1.latitude as latitude_from',
            'pl2.address as address_to',
            'pl2.longitude as longitude_to',
            'pl2.latitude as latitude_to',
            DB::raw($subqueryRides),
            DB::raw($subqueryDistanceFrom),
            DB::raw($subqueryDistanceTo)
        )
            ->join('carpool_pickup_locations as pl1', function ($join) use ($request, $minutesDelta) {
                $requestedRideTime = (float) Carbon::createFromFormat("H:i", $request->time)->format("H.i");
                $requestedRideTimeAfterSub = $requestedRideTime - ($minutesDelta / 60);
                $requestedRideTimeAfterAdd = $requestedRideTime + ($minutesDelta / 60);

                // there will be cases when we subtract or add time, it will go beyond 0 and 24
                // Carbon handle this by converting it to next logical time, for example if we add
                // 5 hours in 23:00, Carbon will return 04:00, but we want even after adding or subtracting                
                // time, we still find rides that lies between these logically converted times
                $subqueryRideTimeWhere = "";
                if ($requestedRideTimeAfterAdd < 24 && $requestedRideTimeAfterSub >= 0) {
                    // normal case where even after subtracting and adding delta, time remains within bounds i.e within 00:00 - 23:59
                    $subqueryRideTimeWhere .= "(";
                    $subqueryRideTimeWhere .= "pl1.time_start <= '" . Carbon::createFromFormat("H:i", $request->time)->addMinutes($minutesDelta)->format("H:i") . "'";
                    $subqueryRideTimeWhere .= " and ";
                    $subqueryRideTimeWhere .= "pl1.time_start >= '" . Carbon::createFromFormat("H:i", $request->time)->subMinutes($minutesDelta)->format("H:i") . "'";
                    $subqueryRideTimeWhere .= ")";
                }

                if ($requestedRideTimeAfterSub < 0) {
                    // after subtracting delta, time is less than 00:00
                    $subqueryRideTimeWhere .= "(";
                    $subqueryRideTimeWhere .= "pl1.time_start <= '" . Carbon::createFromFormat("H:i", $request->time)->addMinutes($minutesDelta)->format("H:i") . "'";
                    $subqueryRideTimeWhere .= " and ( ";
                    $subqueryRideTimeWhere .= "pl1.time_start >= '" . Carbon::createFromFormat("H:i", $request->time)->subMinutes($minutesDelta)->format("H:i") . "'";
                    $subqueryRideTimeWhere .= " or ";
                    $subqueryRideTimeWhere .= "pl1.time_start >= '" . "00:00'";
                    $subqueryRideTimeWhere .= ")"; // bracket close for "and" group
                    $subqueryRideTimeWhere .= ")";
                }

                if ($requestedRideTimeAfterAdd >= 24) {
                    // after adding delta, time is greater than 23:59
                    $subqueryRideTimeWhere .= "(";
                    $subqueryRideTimeWhere .= "("; // bracket start for "and" group
                    $subqueryRideTimeWhere .= "pl1.time_start <= '" . "23:59'";
                    $subqueryRideTimeWhere .= " and ";
                    $subqueryRideTimeWhere .= "pl1.time_start >= '" . Carbon::createFromFormat("H:i", $request->time)->subMinutes($minutesDelta)->format("H:i") . "'";
                    $subqueryRideTimeWhere .= ")"; // bracket close for "and" group
                    $subqueryRideTimeWhere .= " or ";
                    $subqueryRideTimeWhere .= "pl1.time_start <= '" . Carbon::createFromFormat("H:i", $request->time)->addMinutes($minutesDelta)->format("H:i") . "'";
                    $subqueryRideTimeWhere .= ")";
                }

                $join->on('carpool_profiles.id', '=', 'pl1.profile_id')
                    ->where('pl1.type', '=', 'pickup')
                    ->whereRaw($subqueryRideTimeWhere);
            })
            ->join('carpool_pickup_locations as pl2', function ($join) {
                $join->on('carpool_profiles.id', '=', 'pl2.profile_id')
                    ->where('pl2.type', '=', 'drop');
            })
            ->where('is_verified', 1)
            ->whereRaw($subqueryRidesWhere)
            ->whereRaw($subqueryDistanceWhereFrom)
            ->whereRaw($subqueryDistanceWhereTo);

        // filter by time
        /*$providers->where(function ($query) use ($request) {
            $query->where(function ($query) use ($request) {
                $query->whereTime('time_start', '<=', Carbon::createFromFormat("H:i", $request->time)->addMinutes(30)->format("H:i"))
                    ->whereTime('time_start', '>=', Carbon::createFromFormat("H:i", $request->time)->subMinutes(30)->format("H:i"));
            })->orWhere(function ($query) use ($request) {
                $query->whereTime('time_return', '<=', Carbon::createFromFormat("H:i", $request->time)->addMinutes(30)->format("H:i"))
                    ->whereTime('time_return', '>=', Carbon::createFromFormat("H:i", $request->time)->subMinutes(30)->format("H:i"));
            });
        });*/

        // check if driver rides on this day
        $dayOfWeek = strtolower(Carbon::now()->setDateFrom(new DateTime($request->date))->shortEnglishDayOfWeek);
        $providers->where('travel_days', 'like', "%$dayOfWeek%");

        $providers->groupBy('carpool_profiles.id')->orderBy('time_start');
        return $providers;
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function rides()
    {
        return $this->hasMany(CarpoolRide::class, 'profile_id');
    }

    public function locations()
    {
        return $this->hasMany(PickupLocation::class, 'profile_id');
    }
}
