<?php

namespace Vtlabs\Carpool\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Core\Models\Setting;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Carpool\Models\CarpoolRide;
use Vtlabs\Carpool\Models\CarpoolProfile;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carpool\Filters\CarpoolRideFilter;
use Illuminate\Validation\ValidationException;
use Vtlabs\Carpool\Http\Resources\CarpoolRideResource;

class CarpoolRideController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'status' => 'sometimes'
        ]);
        $rides = CarpoolRide::filter($request->all(), CarpoolRideFilter::class)->orderBy('ride_on', 'desc');

        return CarpoolRideResource::collection($rides->paginate());
    }

    public function update(CarpoolRide $ride, Request $request)
    {
        $request->validate([
            'status' => 'sometimes|in:accepted,onway,ongoing,complete,cancelled,rejected',
            'date' => 'sometimes|date',
            'time_from' => 'required_with:date|date_format:H:i',
            'time_to' => 'required_with:date|date_format:H:i'
        ]);

        $ride->fill($request->only('date', 'time_from', 'time_to'));
        $ride->setStatus($request->status);
        $ride->save();

        return new CarpoolRideResource($ride->refresh());
    }

    public function store(Request $request)
    {
        $request->validate([
            'profile_id' => 'required|exists:carpool_profiles,id',
            'ride_on' => 'required|date_format:Y-m-d H:i',
            'address_from' => 'required|string',
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'address_to' => 'required|string',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
            'seats' => 'required|integer'
        ]);

        $user = Auth::user();
        $profile = CarpoolProfile::find($request->profile_id);

        $minutesDelta = 60;
        $minutesSetting = Setting::where('key', 'minutes_delta')->first();
        if ($minutesSetting) {
            $minutesDelta = $minutesSetting->value ? (int)$minutesSetting->value : $minutesDelta;
        }

        // calculate ride price
        $ridePrice = $request->seats * $profile->price;

        // check if driver has enough seats left
        $subqueryRides = "CAST(seats - COALESCE((SELECT SUM(seats) from carpool_rides where date(ride_on) = '" . Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->toDateString() . "'"
            . " and time(ride_on) <= '" . Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->addMinutes($minutesDelta)->toTimeString() . "'"
            . " and time(ride_on) >= '" . Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->subMinutes($minutesDelta)->toTimeString() . "'"
            // . " and status in ('pending', 'accepted')"
            . " and profile_id=carpool_profiles.id), 0) as SIGNED) as seats_available";
        $subqueryRidesWhere = "CAST(seats - COALESCE((SELECT SUM(seats) from carpool_rides where date(ride_on) = '" . Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->toDateString() . "'"
            . " and time(ride_on) <= '" . Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->addMinutes($minutesDelta)->toTimeString() . "'"
            . " and time(ride_on) >= '" . Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->subMinutes($minutesDelta)->toTimeString() . "'"
            // . " and status in ('pending', 'accepted')"
            . " and profile_id=carpool_profiles.id), 0) as SIGNED) >= " . $request->seats;

        $seatsLeft = CarpoolProfile::select('*', DB::raw($subqueryRides))->whereRaw($subqueryRidesWhere)->exists();

        /*$alreadyBooked = ProviderProfile::whereHas('rides', function ($query) use ($request) {
            $query->whereDate('ride_on', '<=', Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->toDateString())
                ->whereTime('ride_on', '<=', Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->addMinutes(30)->toTimeString())
                ->whereTime('ride_on', '>=', Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->subMinutes(30)->toTimeString());
        })->where('id', $provider->id)->exists();*/

        if (!$seatsLeft) {
            throw ValidationException::withMessages([
                'seats' => 'Already booked!'
            ]);
        }

        $ride = new CarpoolRide();
        $ride->fill($request->only(['profile_id', 'ride_on', 'address_from', 'latitude_from', 'longitude_from', 'address_to', 'latitude_to', 'longitude_to', 'seats']));
        $ride->user_id = Auth::user()->id;
        $ride->price = $ridePrice;
        $ride->save();

        // event(new NewRide($ride));

        return new CarpoolRideResource($ride->refresh());
    }

    public function getFareEstimate(CarpoolProfile $profile, Request $request)
    {
        $request->validate([
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
            'seats' => 'required|integer'
        ]);

        // calculate ride price
        $ridePrice = $request->seats * $profile->price;

        return response()->json(["price" => $ridePrice]);
    }

    public function cancel(CarpoolRide $ride)
    {
        $ride->setStatus('cancelled');
        $ride->save();

        // event(new UpdateRide($ride));

        return new CarpoolRideResource($ride->refresh());
    }
}
