<?php

namespace Vtlabs\Carpool\Http\Controllers\Api\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Services\UserService;
use Vtlabs\Carpool\Models\CarpoolProfile;
use Vtlabs\Carpool\Models\PickupLocation;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carpool\Http\Resources\Admin\CarpoolProfileAdminResource;

class CarpoolProfileController extends Controller
{
    public function index(Request $request)
    {
        return CarpoolProfileAdminResource::collection(CarpoolProfile::latest()->paginate());
    }

    public function show(CarpoolProfile $profile)
    {
        return new CarpoolProfileAdminResource($profile);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|max:255|unique:users',
            'mobile_number' => 'required|unique:users',
            'password' => 'required|min:6',
            'meta' => 'sometimes|json|nullable',
            'is_verified' => 'required|boolean',
            'vehicle_details' => 'required|json',
            'seats' => 'required|integer',
            'price' => 'required|numeric',
            'travel_days' => 'required|array|in:sun,mon,tue,wed,thu,fri,sat',
            'locations' => 'required|array',
            'locations.*.address' => 'required',
            'locations.*.longitude' => 'required|numeric',
            'locations.*.latitude' => 'required|numeric',
            'locations.*.type' => 'required|in:pickup,drop',
            'locations.*.is_return' => 'required|nullable',
            'locations.*.time_start' => 'sometimes|nullable|date_format:H:i:s'
        ]);

        request()->merge([
            'meta' => $request->meta ? json_decode($request->meta) : null,
            'vehicle_details' => $request->vehicle_details ? json_decode($request->vehicle_details) : null,
            'travel_days' => implode(",", $request->travel_days)
        ]);

        // create user for profile first  
        $userData = collect($request->only(['email', 'mobile_number', 'password', 'name']));
        $userData = $userData->merge(['mobile_verified' => 1, 'roles' => ['customer']]);
        $user = UserService::create($userData);

        $profile = new CarpoolProfile();
        $profile->user_id = $user->id;

        $profile->fill($request->only([
            'meta', 'is_verified', 'vehicle_details', 'seats', 'price', 'travel_days'
        ]));
        $profile->save();

        // handle pickup/drop locations
        if ($request->locations) {
            PickupLocation::where('profile_id', $profile->id)->delete();
            foreach ($request->locations as $location) {
                $pickupLocation = new PickupLocation();
                $pickupLocation->address = $location['address'];
                $pickupLocation->longitude = $location['longitude'];
                $pickupLocation->latitude = $location['latitude'];
                $pickupLocation->type = $location['type'];
                if (isset($location['time_start']) && $location['time_start']) {
                    $pickupLocation->time_start = $location['time_start'];
                }
                $pickupLocation->profile_id = $profile->id;
                $pickupLocation->save();
            }
        }

        return new CarpoolProfileAdminResource($profile->refresh());
    }

    public function update(CarpoolProfile $profile, Request $request)
    {
        $request->validate([
            'meta' => 'sometimes|json|nullable',
            'is_verified' => 'required|boolean',
            'vehicle_details' => 'required|json',
            'seats' => 'required|integer',
            'price' => 'required|numeric',
            'travel_days' => 'required|array|in:sun,mon,tue,wed,thu,fri,sat',
            'locations' => 'required|array',
            'locations.*.address' => 'required',
            'locations.*.longitude' => 'required|numeric',
            'locations.*.latitude' => 'required|numeric',
            'locations.*.type' => 'required|in:pickup,drop',
            'locations.*.time_start' => 'sometimes|nullable|date_format:H:i:s'
        ]);

        request()->merge([
            'meta' => $request->meta ? json_decode($request->meta) : null,
            'vehicle_details' => $request->vehicle_details ? json_decode($request->vehicle_details) : null,
            'travel_days' => implode(",", $request->travel_days)
        ]);

        $profile->fill($request->only([
            'meta', 'is_verified', 'vehicle_details', 'seats', 'price', 'travel_days'
        ]));
        $profile->save();

        // handle pickup/drop locations
        if ($request->locations) {
            PickupLocation::where('profile_id', $profile->id)->delete();
            foreach ($request->locations as $location) {
                $pickupLocation = new PickupLocation();
                $pickupLocation->address = $location['address'];
                $pickupLocation->longitude = $location['longitude'];
                $pickupLocation->latitude = $location['latitude'];
                $pickupLocation->type = $location['type'];
                if (isset($location['time_start']) && $location['time_start']) {
                    $pickupLocation->time_start = $location['time_start'];
                }
                $pickupLocation->profile_id = $profile->id;
                $pickupLocation->save();
            }
        }

        return new CarpoolProfileAdminResource($profile->refresh());
    }


    public function destroy(CarpoolProfile $profile, Request $request)
    {
        Gate::authorize('delete');

        $profile->delete();

        return response()->json([], 204);
    }
}
