<?php

namespace Vtlabs\Carpool\Http\Controllers\Api;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Vtlabs\Core\Models\Setting;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Carpool\Models\CarpoolRide;
use Vtlabs\Carpool\Models\CarpoolRider;
use Vtlabs\Carpool\Models\CarpoolProfile;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carpool\Filters\CarpoolRideFilter;
use Illuminate\Validation\ValidationException;
use Vtlabs\Carpool\Http\Resources\CarpoolRideResource;
use Vtlabs\Carpool\Http\Resources\CarpoolRiderResource;

class CarpoolRideController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'rideon' => 'sometimes',
            'lat_from' => 'required_with:rideon|numeric',
            'long_from' => 'required_with:rideon|numeric',
            'lat_to' => 'required_with:rideon|numeric',
            'long_to' => 'required_with:rideon|numeric',
            'seats' => 'required_with:rideon|numeric'
        ]);

        if($request->rideon) {
            // implies user is searching for a ride
            return CarpoolRideResource::collection(CarpoolRide::search(Auth::user(), $request)->paginate(config('constants.paginate_per_page')));
        }
        
        $rides = CarpoolRide::filter($request->all(), CarpoolRideFilter::class)->orderBy('ride_on', 'desc');

        return CarpoolRideResource::collection($rides->paginate());
    }

    public function update(CarpoolRide $ride, Request $request)
    {
        $request->validate([
            'status' => 'sometimes|in:pending,started,complete,cancelled,rejected',
            'date' => 'sometimes|date',
            'time_from' => 'required_with:date|date_format:H:i',
            'time_to' => 'required_with:date|date_format:H:i'
        ]);

        $ride->fill($request->only('date', 'time_from', 'time_to'));
        
        if($request->status) {
            $ride->setStatus($request->status);
        }
        
        $ride->save();

        return new CarpoolRideResource($ride->refresh());
    }

    public function store(Request $request)
    {
        $request->validate([
            'profile_id' => 'required|exists:carpool_profiles,id',
            'ride_on' => 'required|date_format:Y-m-d H:i',
            'address_from' => 'required|string',
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'address_to' => 'required|string',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
            'seats' => 'required|numeric',
            'price_per_seat' => 'required|numeric'
        ]);

        $user = Auth::user();
        $profile = CarpoolProfile::find($request->profile_id);

        // $minutesDelta = 60;
        // $minutesSetting = Setting::where('key', 'minutes_delta')->first();
        // if ($minutesSetting) {
        //     $minutesDelta = $minutesSetting->value ? (int)$minutesSetting->value : $minutesDelta;
        // }

        // calculate ride price
        // $ridePrice = $request->seats * $profile->price;

        // check if driver has enough seats left
        // $subqueryRides = "CAST(seats-seats_confirmed) as SIGNED) as seats_available";
        // $subqueryRidesWhere = "seats_available >=" . $request->seats;

        // $seatsLeft = CarpoolProfile::select('*', DB::raw($subqueryRides))->whereRaw($subqueryRidesWhere)->exists();

        /*$alreadyBooked = ProviderProfile::whereHas('rides', function ($query) use ($request) {
            $query->whereDate('ride_on', '<=', Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->toDateString())
                ->whereTime('ride_on', '<=', Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->addMinutes(30)->toTimeString())
                ->whereTime('ride_on', '>=', Carbon::createFromFormat("Y-m-d H:i", $request->ride_on)->subMinutes(30)->toTimeString());
        })->where('id', $provider->id)->exists();*/

        // if (!$seatsLeft) {
        //     throw ValidationException::withMessages([
        //         'seats' => 'Already booked!'
        //     ]);
        // }

        $ride = new CarpoolRide();
        $ride->fill($request->only(['profile_id', 'ride_on', 'address_from', 'latitude_from', 'longitude_from', 'address_to', 'latitude_to', 'longitude_to', 'seats', 'price_per_seat']));
        $ride->seats_left = $ride->seats;
        $ride->save();

        // event(new NewRide($ride));

        return new CarpoolRideResource($ride->refresh());
    }

    public function book(CarpoolRide $ride, Request $request)
    {
        $request->validate([
            'address_from' => 'required|string',
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'address_to' => 'required|string',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
            'seats' => 'required|integer',
            'rideon' => 'required'
        ]);

        $price = $ride->price_per_seat * $request->seats;

        $user = Auth::user();
        if (!$user->canWithdraw($price)) {
            return response()->json(['message' => 'Wallet balance not enough'], 400);
        }

        $request->merge([
            'user_id' => Auth::id(),
            'price' => $price,
            'ride_id' => $ride->id
        ]);

        if(CarpoolRider::where('ride_id', $ride->id)->where('user_id', Auth::id())->exists()) {
            throw ValidationException::withMessages([
                'ride' => 'Already requested for the ride'
            ]);
        }

        $rider = CarpoolRider::create($request->only(['address_from', 'latitude_from', 'longitude_from', 'address_to', 'latitude_to', 'longitude_to', 'seats', 'user_id', 'price', 'ride_id']));

        $ride->profile->user->sendPushNotification(
            'customer',
            sprintf("New request for Ride #%d", $ride->id),
            sprintf("New request for Ride #%d", $ride->id),
        );

        return new CarpoolRiderResource($rider);
    }

    public function updateRider(CarpoolRider $rider, Request $request)
    {
        $request->validate([
            'status' => 'required'
        ]);

        $rider->setStatus($request->status);
        $rider->save();

        if($request->status == 'dropped') {
            $this->dropCustomer($rider);
        }

        if($request->status == 'accepted') {
            $ride = $rider->ride;
            $ride->seats_confirmed += $rider->seats;
            $ride->seats_left -= $rider->seats;
            $ride->save();
        }

        $this->sendUpdateRequestPushNotification($rider, $request->status);

        return new CarpoolRiderResource($rider);
    }

    public function getFareEstimate(CarpoolProfile $profile, Request $request)
    {
        $request->validate([
            'latitude_from' => 'required|numeric',
            'longitude_from' => 'required|numeric',
            'latitude_to' => 'required|numeric',
            'longitude_to' => 'required|numeric',
            'seats' => 'required|integer'
        ]);

        // calculate ride price
        $ridePrice = $request->seats * $profile->price;

        return response()->json(["price" => $ridePrice]);
    }

    public function cancel(CarpoolRide $ride)
    {
        $ride->setStatus('cancelled');
        $ride->save();

        // event(new UpdateRide($ride));

        return new CarpoolRideResource($ride->refresh());
    }

    private function dropCustomer($rider)
    {
        $price = $rider->price;

        $rider->user->forceWithdraw($rider->price, 'withdraw', [
            'description' => 'Paid for ride #' . $rider->ride->id,
            'type' => 'ride_payment'
        ]);

        $adminShare = floatval(Setting::findByKey('admin_commision_value'));
        $ownerEarnings = $price - ($price * $adminShare) / 100;
        $adminTotalEarning = ($price * $adminShare) / 100; // admin gets it's share of earning

        $rider->ride->profile->user->deposit($ownerEarnings, 'deposit', [
            'description' => 'Received for ride #' . $rider->ride->id,
            'type' => 'ride_payment'
        ]);

        User::find(1)->deposit($adminTotalEarning, 'deposit', [
            'description' => 'Admin Commission for ride #' . $rider->ride->id,
            'type' => 'ride_payment'
        ]);
    }

    private function sendUpdateRequestPushNotification($rider, $status) 
    {
        $ride = $rider->ride;

        // handle push notification
        if(in_array($status, ['accepted', 'rejected'])) {
            $rider->user->sendPushNotification(
                'customer',
                sprintf("Your request for Ride #%d is %s", $ride->id, $status),
                sprintf("Your request for Ride #%d is %s", $ride->id, $status),
            );
        }

        if(in_array($status, ['picked', 'dropped'])) {
            $rider->user->sendPushNotification(
                'customer',
                sprintf("%s for Ride #%d", $status, $ride->id),
                sprintf("%s for Ride #%d", $status, $ride->id),
            );
        }
    }
}
