<?php

namespace Vtlabs\Carpool\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Carpool\Models\CarpoolProfile;
use Vtlabs\Carpool\Models\PickupLocation;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Carpool\Http\Resources\CarpoolProfileResource;
use Vtlabs\Carpool\Http\Resources\CarpoolProfileSearchResource;

/**
 * @group Carpool
 *
 * APIs for CARPOOLs
 */
class CarpoolProfileController extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'lat_from' => 'required|numeric',
            'long_from' => 'required|numeric',
            'lat_to' => 'required|numeric',
            'long_to' => 'required|numeric',
            'date' => 'required|date',
            'time' => 'required|date_format:H:i',
            'seats' => 'required|numeric'
        ]);

        return CarpoolProfileSearchResource::collection(CarpoolProfile::search(Auth::user(), $request)->paginate(config('constants.paginate_per_page')));
    }

    public function show(CarpoolProfile $profile)
    {
        return new CarpoolProfileResource($profile);
    }

    public function showMyProfile(Request $request)
    {
        $profile = CarpoolProfile::findByUser(Auth::id());
        return new CarpoolProfileResource($profile);
    }

    public function update(Request $request)
    {
        $request->validate([
            'vehicle_details' => 'sometimes|json',
            'seats' => 'sometimes|integer',
            'price' => 'sometimes|numeric',
            'travel_days' => 'sometimes|array|in:sun,mon,tue,wed,thu,fri,sat',
            'locations' => 'sometimes|array',
            'locations.*.address' => 'required',
            'locations.*.longitude' => 'required|numeric',
            'locations.*.latitude' => 'required|numeric',
            'locations.*.type' => 'required|in:pickup,drop',
            'locations.*.time_start' => 'required|date_format:H:i'
        ]);

        if ($request->meta) {
            request()->merge(['meta' => $request->meta ? json_decode($request->meta) : null]);
        }

        if ($request->vehicle_details) {
            request()->merge(['vehicle_details' => $request->vehicle_details ? json_decode($request->vehicle_details) : null]);
        }

        if ($request->travel_days) {
            request()->merge(['travel_days' => implode(",", $request->travel_days)]);
        }

        $requestData = $request->only([
            'vehicle_details', 'seats', 'price', 'meta', 'travel_days'
        ]);

        try {
            $profile = CarpoolProfile::findByUser(Auth::id());
        } catch (\Exception $ex) {
            $profile = new CarpoolProfile();
            $profile->user_id = Auth::id();
        }

        $profile->fill($requestData);
        $profile->save();

        // handle pickup/drop locations
        if ($request->locations) {
            PickupLocation::where('profile_id', $profile->id)->delete();
            foreach ($request->locations as $location) {
                $pickupLocation = new PickupLocation();
                $pickupLocation->address = $location['address'];
                $pickupLocation->longitude = $location['longitude'];
                $pickupLocation->latitude = $location['latitude'];
                $pickupLocation->type = $location['type'];
                if (isset($location['time_start']) && $location['time_start']) {
                    $pickupLocation->time_start = $location['time_start'];
                }
                $pickupLocation->profile_id = $profile->id;
                $pickupLocation->save();
            }
        }

        return new CarpoolProfileResource($profile->refresh());
    }
}
