<?php

namespace Vtlabs\Carpool\Models;

use Carbon\Carbon;
use Illuminate\Http\Request;
use EloquentFilter\Filterable;
use Vtlabs\Core\Models\Setting;
use Vtlabs\Core\Models\User\User;
use Illuminate\Support\Facades\DB;
use Spatie\ModelStatus\HasStatuses;
use Vtlabs\Payment\Traits\CanBePaid;
use Vtlabs\Payment\Contracts\Payable;
use Illuminate\Database\Eloquent\Model;
use Vtlabs\Carpool\Models\CarpoolRider;

class CarpoolRide extends Model implements Payable
{
    use CanBePaid, HasStatuses, Filterable;

    protected $table = 'carpool_rides';

    protected $guarded = [];

    public static function boot()
    {
        parent::boot();

        // on create
        static::created(function ($order) {
            $order->setStatus(config('vtlabs_carpool.ride.status_default'));
        });
    }

    public static function search($user, Request $request)
    {
        $distanceDelta = 10000000;
        $minutesDelta = 60;

        $distanceSetting = Setting::where('key', 'distance_limit')->first();
        if ($distanceSetting) {
            $distanceDelta = $distanceSetting->value ? (int) $distanceSetting->value : $distanceDelta;
        }

        $minutesSetting = Setting::where('key', 'minutes_delta')->first();
        if ($minutesSetting) {
            $minutesDelta = $minutesSetting->value ? (int) $minutesSetting->value : $minutesDelta;
        }

        // show providers listed with in particular distance
        $subqueryDistanceFrom = "ST_Distance_Sphere(Point(longitude_from,"
            . " latitude_from),"
            . " Point($request->long_from, $request->lat_from ))"
            . " as distance_from";
        $subqueryDistanceTo = "ST_Distance_Sphere(Point(longitude_to,"
            . " latitude_to),"
            . " Point($request->long_to, $request->lat_to))"
            . " as distance_to";

        $subqueryDistanceWhereFrom = "ST_Distance_Sphere(Point(longitude_from,"
            . " latitude_from),"
            . " Point($request->long_from, $request->lat_from ))"
            . " < " . $distanceDelta;

        $subqueryDistanceWhereTo = "ST_Distance_Sphere(Point(longitude_to,"
            . " latitude_to),"
            . " Point($request->long_to, $request->lat_to ))"
            . " < " . $distanceDelta;

        // check if driver has enough seats left
        $subqueryRidesWhere = "seats_left >=" . $request->seats;

        $providers = CarpoolRide::select(
            'carpool_rides.*',
            DB::raw($subqueryDistanceFrom),
            DB::raw($subqueryDistanceTo)
        )
            ->whereRaw($subqueryRidesWhere)
            ->whereRaw($subqueryDistanceWhereFrom)
            ->whereRaw($subqueryDistanceWhereTo);

        $providers =    $providers->whereDate('ride_on', '=', Carbon::createFromFormat("Y-m-d H:i", $request->rideon)->toDateString())
            ->whereTime('ride_on', '<=', Carbon::createFromFormat("Y-m-d H:i", $request->rideon)->addMinutes(30)->toTimeString())
            ->whereTime('ride_on', '>=', Carbon::createFromFormat("Y-m-d H:i", $request->rideon)->subMinutes(30)->toTimeString());

        $providers = $providers->currentStatus('pending');

        $providers = $providers->where('is_verified', 1);

        return $providers;
    }

    public function isValidStatus(string $name, ?string $reason = null): bool
    {
        return in_array($name, config('vtlabs_carpool.ride.status_list', []));
    }

    public function profile()
    {
        return $this->belongsTo(CarpoolProfile::class, 'profile_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function riders()
    {
        return $this->hasMany(CarpoolRider::class, 'ride_id');
    }
}
