<?php

namespace Vtlabs\Cardgame\Models;

use Carbon\Carbon;
use EloquentFilter\Filterable;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Database\Eloquent\Model;
use Vtlabs\Cardgame\Models\Pokerplayer;
use Vtlabs\Core\Services\FirebaseService;
use Vtlabs\Cardgame\Jobs\Poker\StartPoker;

class Poker extends Model
{
    use Filterable;

    protected $table = 'cardgame_pokers';

    protected $guarded = [];

    protected $casts = [
        'meta' => 'json',
        'max_turn_amount' => 'float',
        'pot_value' => 'float',
        'player_turn' => 'float',
        'last_player_turn' => 'float',
        'start_at' => 'datetime', // UTC ISO-8601
        'first_player_request_at' => 'datetime'
    ];

    protected $with = ['pokerplayers'];

    public static function findPoker(Profile $profile, float $turnAmount, float $joiningBalance)
    {
        $settings = CoreHelper::settingsAsDictionary();
        $pokers = Poker::where('status', 'starting')->where('start_at', '>=', Carbon::now()->addSeconds($settings['stop_new_entry_in_seconds']));

        if ($pokers->exists()) {
            $poker =  $pokers->first();

            // $maxTurnAmount = $poker->max_turn_amount >= $turnAmount ? $turnAmount : $poker->max_turn_amount;

            $poker->pokerplayers()->save(new Pokerplayer([
                'status' => 'waiting',
                //'turn_amount' => $maxTurnAmount,
                'joining_balance' => $joiningBalance,
                'profile_id' => $profile->id
            ]));
        } else {
            // poker is created and first user is waiting, that means this will be the second user of the game i.e big blind of the game, in this case, turn amount of game
            // should be equal to the turn amount expected by this player
            $pokers = Poker::where('status', 'pending')->where('first_player_request_at', '>=', Carbon::now()->subSeconds($settings['first_player_waiting_in_seconds']))
                ->where('max_turn_amount', $turnAmount);
            if ($pokers->exists()) {
                $poker =  $pokers->first();

                $poker->start_at = Carbon::now()->addSeconds($settings['new_game_starting_in_seconds']);
                $poker->status = 'starting';
                $poker->save();

                $poker->pokerplayers()->save(new Pokerplayer([
                    'status' => 'waiting',
                    'turn_amount' => $turnAmount,
                    'joining_balance' => $joiningBalance,
                    'profile_id' => $profile->id
                ]));

                // dispatch a job which will start the poker
                StartPoker::dispatch($poker->id)->delay(now()->addSeconds($settings['new_game_starting_in_seconds']));
            } else {
                // poker is created but is in stalled state
                $pokers = Poker::where('status', 'pending')->where('first_player_request_at', '<', Carbon::now()->subSeconds($settings['first_player_waiting_in_seconds']));
                if ($pokers->exists()) {
                    $poker =  $pokers->first();

                    // reset poker
                    $poker->max_turn_amount = $turnAmount;
                    $poker->pot_value = 0;
                    $poker->first_player_request_at = Carbon::now();
                    $poker->save();

                    // delete old players and insert current player
                    $poker->pokerplayers()->delete();
                    $poker->pokerplayers()->save(new Pokerplayer([
                        'status' => 'waiting',
                        'turn_amount' => $turnAmount/2,
                        'joining_balance' => $joiningBalance,
                        'profile_id' => $profile->id
                    ]));
                } else {
                    $poker = self::generatePoker($profile, $turnAmount, $joiningBalance);
                }
            }
        }

        $poker = $poker->fresh();

        $firebaseDatabase = FirebaseService::getDatabaseInstance();
        if ($firebaseDatabase) {
            $firebaseDatabase->getReference('/fire_app/poker/' . $poker->id)->set($poker);
        }

        return $poker;
    }

    private static function generatePoker(Profile $profile, float $smallBlindTurnAmount, float $smallBlindJoiningBalance)
    {
        $poker = Poker::create([
            'max_turn_amount' => $smallBlindTurnAmount,
            'first_player_request_at' => Carbon::now()
        ]);

        $poker->pokerplayers()->save(new Pokerplayer([
            'status' => 'waiting',
            'turn_amount' => $smallBlindTurnAmount/2,
            'joining_balance' => $smallBlindJoiningBalance,
            'profile_id' => $profile->id
        ]));

        return $poker;
    }

    public function pokerplayers()
    {
        return $this->hasMany(Pokerplayer::class, 'poker_id')->oldest();
    }

    public function nextStatus()
    {
        if($this->status == 'preflop') {
            return 'flop';
        }

        if($this->status == 'flop') {
            return 'turn';
        }

        if($this->status == 'turn') {
            return 'river';
        }

        if($this->status == 'river') {
            return 'showdown';
        }

        return false;
    }
}
