<?php

namespace Vtlabs\Cardgame\Models;

use Carbon\Carbon;
use EloquentFilter\Filterable;
use Vtlabs\Cardgame\Jobs\StartGame;
use Vtlabs\Core\Helpers\CoreHelper;
use Vtlabs\Cardgame\Models\Gameplayer;
use Illuminate\Database\Eloquent\Model;
use Vtlabs\Core\Services\FirebaseService;

class Game extends Model
{
    use Filterable;

    const ACTIVE_GAME_STATUS = ['starting', 'started', 'tossed', 'toss_arranged', 'toss_unassigned', 'cards_spread', 'turn_setup'];

    protected $table = 'cardgame_games';

    protected $guarded = [];

    protected $casts = [
        'meta' => 'json',
        'start_at' => 'datetime', // UTC ISO-8601
        'first_player_request_at' => 'datetime'
    ];

    protected $with = ['gameplayers', 'cashtable'];

    public static function findGame(Profile $profile, $cashtableId = null)
    {
        $settings = CoreHelper::settingsAsDictionary();
        $games = Game::where('status', 'starting')->where('start_at', '>=', Carbon::now()->addSeconds($settings['stop_new_entry_in_seconds']));

        if ($cashtableId) {
            $games->where('cashtable_id', $cashtableId);
        }

        if ($games->exists()) {
            $game =  $games->first();

            $game->gameplayers()->save(new Gameplayer([
                'status' => 'waiting',
                'profile_id' => $profile->id
            ]));
        } else {
            // game is created and first user is waiting
            $games = Game::where('status', 'pending')->where('first_player_request_at', '>=', Carbon::now()->subSeconds($settings['first_player_waiting_in_seconds']));

            if ($cashtableId) {
                $games->where('cashtable_id', $cashtableId);
            }

            if ($games->exists()) {
                $game =  $games->first();

                $game->start_at = Carbon::now()->addSeconds($settings['new_game_starting_in_seconds']);
                $game->status = 'starting';
                $game->save();

                $game->gameplayers()->save(new Gameplayer([
                    'status' => 'waiting',
                    'profile_id' => $profile->id
                ]));

                // dispatch a job which will start the game
                StartGame::dispatch($game->id)->delay(now()->addSeconds($settings['new_game_starting_in_seconds']));
            } else {
                // game is created but is in stalled state
                $games = Game::where('status', 'pending')->where('first_player_request_at', '<', Carbon::now()->subSeconds($settings['first_player_waiting_in_seconds']));

                if ($cashtableId) {
                    $games->where('cashtable_id', $cashtableId);
                }

                if ($games->exists()) {
                    $game =  $games->first();

                    // reset game timings
                    $game->first_player_request_at = Carbon::now();
                    $game->next_turn_uuid = null;
                    $game->save();

                    // delete old players and insert current player
                    $game->gameplayers()->delete();
                    $game->gameplayers()->save(new Gameplayer([
                        'status' => 'waiting',
                        'profile_id' => $profile->id
                    ]));
                } else {
                    $game = self::generateGame($profile, $cashtableId);
                }
            }
        }

        $game = $game->fresh();

        $firebaseDatabase = FirebaseService::getDatabaseInstance();
        if ($firebaseDatabase) {
            $firebaseDatabase->getReference('/fire_app/games/' . $game->id)->set($game);
        }

        return $game;
    }

    public static function activePlayersByCashtableCount($cashtableId)
    {

        $totalActivePlayers = GamePlayer::whereHas('game', function ($query) use ($cashtableId) {
            $query->where('cashtable_id', $cashtableId)->whereIn('status', Game::ACTIVE_GAME_STATUS);
        })->whereIn('status', ['active', 'waiting'])->count();

        return $totalActivePlayers;
    }

    public function findPlayerByPosition($position)
    {
        return $this->gameplayers->where('position', $position)->first();
    }

    public function gameplayers()
    {
        return $this->hasMany(Gameplayer::class, 'game_id');
    }

    public function cashtable()
    {
        return $this->belongsTo(Cashtable::class, 'cashtable_id');
    }

    public function calculatePoints()
    {
        // on game complete, find the the player with lowest points take the sum of points of all other players minus the points of the player with lowest points
        $winner = $this->gamePlayers->sortBy('points')->first();
        $losers = $this->gamePlayers->where('id', '!=', $winner->id);

        $cashTableAmount = $this->cashtable->reward;

        $totalLoserPoints = $losers->sum('points');
        $winnerPoints = $winner->points;
        $winnerAmount = ($totalLoserPoints - $winnerPoints) * $cashTableAmount;

        // Update winner's wallet (credit)
        $winner->profile->user->deposit($winnerAmount, 'deposit', [
            'description' => 'Winning earnings deposited',
            'type' => 'earnings',
            'source' => 'earnings',
            'source_id' => $this->id,
        ]);

        $winnerProfile = $winner->profile;
        $winnerProfile->balance = $winnerProfile->balance + $winnerAmount;
        $winnerProfile->save();

        // Update losers' wallets (debit)
        foreach ($losers as $loser) {
            $loserAmount = $loser->points * $cashTableAmount;
            $loser->profile->user->forceWithdraw($loserAmount, 'withdraw', [
                'description' => 'Lost amount withdrawl',
                'type' => 'loss',
                'source' => 'loss',
                'source_id' => $this->id,
            ]);
            $loserProfile = $loser->profile;
            $loserProfile->balance = $loserProfile->balance - $loserAmount;
            $loserProfile->save();
        }
    }

    public function setupNextTurn()
    {
        $lastPlayer = $this->player_turn;
        $totalPlayerCount = $this->gamePlayers()->count();
        $nextPlayer = $this->player_turn < $totalPlayerCount ? $this->player_turn + 1 : 1;

        while (1) {
            $player = $this->findPlayerByPosition($nextPlayer);
            if ($player->status == 'active') {
                break;
            }

            if ($lastPlayer == $nextPlayer) {
                throw new \Exception('No active player found');
            }

            $nextPlayer = $nextPlayer < $this->gamePlayers()->count() ? $nextPlayer + 1 : 1;
        }

        $this->player_turn = $nextPlayer;
        $this->last_player_turn = $lastPlayer;

        return $this;
    }

    private static function generateGame(Profile $profile, $cashtableId = null)
    {
        $game = Game::create([
            'cashtable_id' => $cashtableId,
            'first_player_request_at' => Carbon::now()
        ]);

        $game->gameplayers()->save(new Gameplayer([
            'status' => 'waiting',
            'profile_id' => $profile->id
        ]));

        return $game;
    }
}
