<?php

namespace Vtlabs\Cardgame\Jobs;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Illuminate\Bus\Queueable;
use Vtlabs\Cardgame\Models\Game;
use Illuminate\Support\Facades\Log;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Queue\SerializesModels;
use Illuminate\Queue\InteractsWithQueue;
use Vtlabs\Core\Services\FirebaseService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class AutoTurn implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    private $gameId;
    private $expectedPlayerTurn;
    private $nextTurnUuid;

    public function __construct($gameId, $expectedPlayerTurn, $nextTurnUuid)
    {
        $this->gameId = $gameId;
        $this->expectedPlayerTurn = $expectedPlayerTurn;
        $this->nextTurnUuid = $nextTurnUuid;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        try {

            Log::info("Running auto turn in game: " . $this->gameId . " for player: " . $this->expectedPlayerTurn);

            $game = Game::find($this->gameId);

            if ($game->next_turn_uuid != $this->nextTurnUuid) {
                Log::info("Skipping auto turn as next turn uuid does not match");
                $this->fail('Auto turn not required');
                return;
            }

            Log::info("Running auto turn - player: " . $this->expectedPlayerTurn . " has skipped his turn");
            

            $player = $game->findPlayerByPosition($this->expectedPlayerTurn);
            $player->empty_turns = $player->empty_turns + 1;

            if ($player->empty_turns >= 3) {
                $player->status = 'dropped';
                $player->points = 40;
            }

            $player->save();

            $game = $game->setupNextTurn();

            $game->next_turn_uuid = Str::uuid();

            $game->save();

            $firebaseDatabase = FirebaseService::getDatabaseInstance();
            if ($firebaseDatabase) {
                $firebaseDatabase->getReference('/fire_app/games/' . $game->id)->set($game);
            }

            // run next job which will handle the auto turn
            $settings = CoreHelper::settingsAsDictionary();
            AutoTurn::dispatch($game->id, $game->player_turn, $game->next_turn_uuid)->delay(now()->addSeconds($settings['turn_interval_in_seconds']));
        } catch (\Exception $ex) {
            //
        }
    }
}
