<?php

namespace Vtlabs\Cardgame\Http\Controllers\Api;

use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Vtlabs\Cardgame\Models\Game;
use Vtlabs\Cardgame\Jobs\AutoTurn;
use Illuminate\Support\Facades\Log;
use Vtlabs\Cardgame\Models\Profile;
use Vtlabs\Core\Helpers\CoreHelper;
use Illuminate\Support\Facades\Auth;
use Vtlabs\Cardgame\Models\Gameplayer;
use Vtlabs\Core\Services\FirebaseService;
use Vtlabs\Core\Http\Controllers\Controller;

class GameController extends Controller
{
    public function findGame(Request $request)
    {
        $request->validate([
            'cashtable_id' => 'sometimes'
        ]);
        $profile = Profile::findByUser(Auth::id());

        $game = Game::findGame($profile, $request->cashtable_id);

        return response()->json($game);
    }

    public function playTurn(Game $game, Gameplayer $gameplayer, Request $request)
    {
        $request->validate([
            'picked_card' => 'required',
            'picked_deck' => 'required|in:open,close',
            'discarded_card' => 'required'
        ]);

        $cards = explode(',', $game->cards);
        $openCards = explode(',', $game->open_cards);

        // pick card from open/close deck
        if($request->picked_deck == 'open') {
            array_pop($openCards);
        } else {
            array_pop($cards);
        }

        // give card to player
        $playerCards = explode(',', $gameplayer->cards);
        array_push($playerCards, $request->picked_card);
        

        // remove discarded card from player cards
        $playerCards = array_values(array_diff($playerCards, [$request->discarded_card]));

        // add discarded card to open deck
        array_push($openCards, $request->discarded_card);
        $game->open_cards = implode(',', $openCards);

        // next player turn
        $game = $game->setupNextTurn();
        
        $gameplayer->cards = implode(',', $playerCards);
        $gameplayer->turn_count = $gameplayer->turn_count + 1;
        $gameplayer->empty_turns = 0;
        $game->cards = implode(',', $cards);
        $game->next_turn_uuid = Str::uuid();

        $gameplayer->save();
        $game->save();

        $game = $game->fresh();

        $this->writeOnFirebase($game);

        // monitor next player turn for auto turn
        $settings = CoreHelper::settingsAsDictionary();
        AutoTurn::dispatch($game->id, $game->player_turn, $game->next_turn_uuid)->delay(now()->addSeconds($settings['turn_interval_in_seconds']));

        return response()->json($game);
    }

    public function declare(Game $game, Gameplayer $gameplayer, Request $request)
    {
        $request->validate([
            'status' => 'required|in:dropped,declared',
            'discarded_card' => 'sometimes',
            'arranged_cards' => 'required|json',
            'points' => 'required'
        ]);

        $cards = explode(',', $game->cards);
        $openCards = explode(',', $game->open_cards);
        $playerCards = explode(',', $gameplayer->cards);

        if($request->discarded_card) {
            $playerCards = array_values(array_diff($playerCards, [$request->discarded_card]));
            $gameplayer->cards = implode(',', $playerCards);

            array_push($openCards, $request->discarded_card);
            $game->open_cards = implode(',', $openCards);
        }

        $gameplayer->status = $request->status;
        $gameplayer->points = $request->points;

        $gameplayer->save();

        // check if there is any active player in the game
        $activePlayerCount = 0;
        foreach ($game->gamePlayers as $gp) {
            if($gp->id != $gameplayer->id && $gp->status == 'active') {
                $activePlayerCount++;
            }
        }

        if($activePlayerCount == 0) {
            $game->status = 'complete';
            $game->save();
            $game = $game->fresh();

            $game->calculatePoints();
            $game = $game->fresh();
        } else {
            $game = $game->fresh();
        }

        $this->writeOnFirebase($game);

        return response()->json($game);
    }

    public function pickCard(Game $game, Gameplayer $gameplayer, Request $request)
    {
        $gameplayer->is_card_picked = 1;
        $gameplayer->save();

        $this->writeOnFirebase($game);
        
        return response()->json($game);
    }

    private function writeOnFirebase(Game $game)
    {
        // update on firebase
        $firebaseDatabase = FirebaseService::getDatabaseInstance();
        if ($firebaseDatabase) {
            $firebaseDatabase->getReference('/fire_app/games/' . $game->id)->set($game);
        }
    }
}
