<?php

namespace Vtlabs\Cardgame\Http\Controllers\Api\Admin;

use Vtlabs\Cardgame\Models\Level;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Vtlabs\Core\Http\Controllers\Controller;
use Vtlabs\Cardgame\Http\Resources\Admin\LevelAdminResource;
use Vtlabs\Core\Helpers\CoreHelper;

class LevelController extends Controller
{
    public function index(Request $request)
    {
        $levels = Level::ordered();

        if($request->parent == '1') {
            $levels = $levels->doesntHave('parent');
        }

        if($request->pagination == '0') {
            $levels = $levels->get();
        } else {
            $levels = $levels->paginate();
        }

        return LevelAdminResource::collection($levels);
    }

    public function show(Level $level)
    {
        return new LevelAdminResource($level);
    }

    public function store(Request $request)
    {
        $request->validate($this->rules());

        CoreHelper::requestMergeJsonField();

        $level = Level::create($request->only($this->fillableFields()));

        if ($request->rewards) {
            $level->rewards()->createMany($request->rewards);
        }

        CoreHelper::requestSaveImage($level, 'image');

        return new LevelAdminResource($level->refresh());
    }

    public function update(Request $request, Level $level)
    {
        $request->validate($this->rules());

        CoreHelper::requestMergeJsonField();

        $level
            ->fill($request->only($this->fillableFields()))
            ->save();

        if ($request->rewards) {
            $level->rewards()->delete();
            $level->rewards()->createMany($request->rewards);
        }

        CoreHelper::requestSaveImage($level, 'image', true);

        return new LevelAdminResource($level->refresh());
    }

    public function destroy(Level $level)
    {
        Gate::authorize('delete');

        $level->delete();

        return response()->json([], 204);
    }

    private function rules(): array
    {
        return [
            'title_translations' => 'required|json|translation',
            'condition_type' => 'required',
            'condition_count' => 'required',
            'sort_order' => 'required',
            'parent_id' => 'sometimes|nullable',
            'meta' => 'sometimes|json|nullable',
            'image' => 'sometimes|image',
            'rewards.*' => 'sometimes|array',
            'rewards.*.type' => 'required',
            'rewards.*.quantity' => 'required',
            'rewards.*.product_id' => 'sometimes|nullable',
        ];
    }

    private function fillableFields(): array
    {
        return ['title', 'condition_type', 'condition_count', 'sort_order', 'parent_id', 'meta'];
    }
}
